<?php
/**
 * Venue - Revenue Analytics
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireVenue();

// Check permission to view analytics
if (!hasPermission('venue.analytics.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/venue/dashboard.php?error=' . urlencode('You do not have permission to view analytics.'));
    exit;
}

$venue = getCurrentVenue();
$pdo = getDB();

// Get revenue statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT e.id) as total_events,
        SUM(b.final_amount) as total_revenue,
        AVG(b.final_amount) as avg_event_revenue,
        COUNT(DISTINCT b.id) as total_bookings
    FROM events e
    LEFT JOIN bookings b ON e.id = b.event_id AND b.payment_status = 'completed'
    WHERE e.venue_id = :venue_id
");
$stmt->execute([':venue_id' => $venue['id']]);
$revenue_stats = $stmt->fetch();

// Get events by genre/type
$stmt = $pdo->prepare("
    SELECT 
        e.event_type,
        COUNT(*) as event_count,
        SUM(b.final_amount) as revenue
    FROM events e
    LEFT JOIN bookings b ON e.id = b.event_id AND b.payment_status = 'completed'
    WHERE e.venue_id = :venue_id
    GROUP BY e.event_type
    ORDER BY revenue DESC
");
$stmt->execute([':venue_id' => $venue['id']]);
$events_by_type = $stmt->fetchAll();

// Get peak times
$stmt = $pdo->prepare("
    SELECT 
        DATE_FORMAT(e.start_date, '%Y-%m') as month,
        COUNT(*) as event_count,
        SUM(b.final_amount) as revenue
    FROM events e
    LEFT JOIN bookings b ON e.id = b.event_id AND b.payment_status = 'completed'
    WHERE e.venue_id = :venue_id
    GROUP BY DATE_FORMAT(e.start_date, '%Y-%m')
    ORDER BY month DESC
    LIMIT 12
");
$stmt->execute([':venue_id' => $venue['id']]);
$peak_times = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Revenue Analytics | Venue Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="venue-panel">
    <?php 
    $current_page = 'analytics';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Revenue Analytics</h1>
                
                <!-- Statistics -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">📅</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($revenue_stats['total_events'] ?? 0); ?></div>
                            <div class="stat-label">Total Events</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">💰</div>
                        <div class="stat-info">
                            <div class="stat-value">$<?php echo number_format($revenue_stats['total_revenue'] ?? 0, 2); ?></div>
                            <div class="stat-label">Total Revenue</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">📊</div>
                        <div class="stat-info">
                            <div class="stat-value">$<?php echo number_format($revenue_stats['avg_event_revenue'] ?? 0, 2); ?></div>
                            <div class="stat-label">Avg Event Revenue</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">🎟️</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($revenue_stats['total_bookings'] ?? 0); ?></div>
                            <div class="stat-label">Total Bookings</div>
                        </div>
                    </div>
                </div>
                
                <!-- Events by Type -->
                <div class="section">
                    <h2>Performance by Event Type</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Event Type</th>
                                    <th>Event Count</th>
                                    <th>Revenue</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($events_by_type)): ?>
                                <tr>
                                    <td colspan="3" style="text-align: center; padding: 2rem;">
                                        No data available yet.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($events_by_type as $type): ?>
                                    <tr>
                                        <td><?php echo ucfirst($type['event_type'] ?? 'N/A'); ?></td>
                                        <td><?php echo $type['event_count']; ?></td>
                                        <td>$<?php echo number_format($type['revenue'] ?? 0, 2); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- Peak Times -->
                <div class="section">
                    <h2>Peak Time Analysis</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Month</th>
                                    <th>Event Count</th>
                                    <th>Revenue</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($peak_times)): ?>
                                <tr>
                                    <td colspan="3" style="text-align: center; padding: 2rem;">
                                        No data available yet.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($peak_times as $peak): ?>
                                    <tr>
                                        <td><?php echo date('F Y', strtotime($peak['month'] . '-01')); ?></td>
                                        <td><?php echo $peak['event_count']; ?></td>
                                        <td>$<?php echo number_format($peak['revenue'] ?? 0, 2); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
</body>
</html>

