<?php
/**
 * Venue Dashboard
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/roles.php';
require_once __DIR__ . '/../includes/url_helper.php';

requireVenue();

$venue = getCurrentVenue();
$user = getCurrentUser();

if (!$venue || !isset($venue['id'])) {
    error_log("Venue Dashboard Error: Venue data not found or invalid");
    die("Venue data not found. Please contact administrator.");
}

$pdo = getDB();

// Get statistics
$stats = [];

// Total events at venue
$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM events WHERE venue_id = :venue_id");
$stmt->execute([':venue_id' => $venue['id']]);
$stats['total_events'] = $stmt->fetch()['total'] ?? 0;

// Upcoming events
$stmt = $pdo->prepare("
    SELECT COUNT(*) as total 
    FROM events 
    WHERE venue_id = :venue_id AND start_date >= NOW() AND status IN ('approved', 'live')
");
$stmt->execute([':venue_id' => $venue['id']]);
$stats['upcoming_events'] = $stmt->fetch()['total'];

// Total occupancy
$stmt = $pdo->prepare("
    SELECT SUM(tt.sold) as total 
    FROM ticket_types tt
    INNER JOIN events e ON tt.event_id = e.id
    WHERE e.venue_id = :venue_id
");
$stmt->execute([':venue_id' => $venue['id']]);
$stats['total_occupancy'] = $stmt->fetch()['total'] ?? 0;

// Utilization rate
$capacity = $venue['capacity'] ?? 0;
$stats['utilization_rate'] = $capacity > 0 ? ($stats['total_occupancy'] / $capacity) * 100 : 0;

// Recent events - Check both venue_id and also events with NULL venue_id if needed
$stmt = $pdo->prepare("
    SELECT e.*, o.company_name as organizer_name
    FROM events e
    LEFT JOIN organizers o ON e.organizer_id = o.id
    WHERE e.venue_id = :venue_id
    ORDER BY e.start_date DESC
    LIMIT 10
");
$stmt->execute([':venue_id' => $venue['id']]);
$recent_events = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Debug: Check what's happening
if (empty($recent_events)) {
    // Check if venue_id exists in events table
    $checkStmt = $pdo->prepare("SELECT COUNT(*) as total FROM events WHERE venue_id = :venue_id");
    $checkStmt->execute([':venue_id' => $venue['id']]);
    $checkResult = $checkStmt->fetch(PDO::FETCH_ASSOC);
    
    error_log("Venue Dashboard Debug: venue_id=" . $venue['id'] . ", venue_name=" . ($venue['venue_name'] ?? 'N/A') . ", events_count=" . ($checkResult['total'] ?? 0));
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Venue Dashboard | beLIVE EventsHUB</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <?php $baseUrl = getCorrectBaseUrl(); ?>
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="venue-panel">
    <?php 
    $current_page = 'dashboard';
    include __DIR__ . '/includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Venue Dashboard</h1>
                <p>Welcome, <?php echo htmlspecialchars($venue['venue_name']); ?>!</p>
                
                <!-- Statistics Cards -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">📅</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($stats['total_events']); ?></div>
                            <div class="stat-label">Total Events</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">⏰</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($stats['upcoming_events']); ?></div>
                            <div class="stat-label">Upcoming Events</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">👥</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($stats['total_occupancy']); ?></div>
                            <div class="stat-label">Total Occupancy</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">📊</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($stats['utilization_rate'], 1); ?>%</div>
                            <div class="stat-label">Utilization Rate</div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Events -->
                <div class="section">
                    <h2>Events at This Venue</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Event Name</th>
                                    <th>Organizer</th>
                                    <th>Start Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($recent_events)): ?>
                                <tr>
                                    <td colspan="5" style="text-align: center; padding: 2rem; color: #6b7280;">
                                        No events found for this venue yet.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($recent_events as $event): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($event['name']); ?></td>
                                        <td><?php echo htmlspecialchars($event['organizer_name'] ?? 'N/A'); ?></td>
                                        <td><?php echo date('M d, Y', strtotime($event['start_date'])); ?></td>
                                        <td><span class="status-badge status-<?php echo $event['status']; ?>"><?php echo ucfirst($event['status']); ?></span></td>
                                        <td>
                                            <a href="pages/event-detail.php?id=<?php echo $event['id']; ?>" class="btn-sm btn-primary">View</a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
</body>
</html>

