<?php
/**
 * Run Roles & Permissions Migration
 * Execute this file once to set up the roles and permissions system
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/auth.php';

// Only allow execution from command line or localhost
$allowed = (
    php_sapi_name() === 'cli' ||
    $_SERVER['REMOTE_ADDR'] === '127.0.0.1' ||
    $_SERVER['REMOTE_ADDR'] === '::1' ||
    (isset($_GET['key']) && $_GET['key'] === 'migrate_roles_2024')
);

if (!$allowed) {
    die('Access denied. This script can only be run from command line or with a valid key.');
}

try {
    $pdo = getDB();
    
    echo "Starting Roles & Permissions Migration...\n\n";
    
    // Read migration file
    $migrationFile = __DIR__ . '/migrations/add_roles_permissions.sql';
    if (!file_exists($migrationFile)) {
        die("Migration file not found: $migrationFile\n");
    }
    
    $sql = file_get_contents($migrationFile);
    
    // Split by semicolon and execute each statement
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && !preg_match('/^--/', $stmt);
        }
    );
    
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    foreach ($statements as $statement) {
        if (empty(trim($statement))) continue;
        
        try {
            $pdo->exec($statement);
            echo "✓ Executed statement\n";
        } catch (PDOException $e) {
            // Ignore "table already exists" errors
            if (strpos($e->getMessage(), 'already exists') === false) {
                echo "⚠ Warning: " . $e->getMessage() . "\n";
            }
        }
    }
    
    // Check if user_type column exists in user_roles table
    $checkColumn = $pdo->query("SHOW COLUMNS FROM user_roles LIKE 'user_type'");
    $hasUserType = $checkColumn->rowCount() > 0;
    
    // Assign default roles to existing admins
    echo "\nAssigning default roles to existing admins...\n";
    
    $stmt = $pdo->query("
        SELECT a.user_id, a.role 
        FROM admins a
        INNER JOIN customers c ON a.user_id = c.id
        WHERE a.is_active = 1 AND c.is_active = 1
    ");
    $admins = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($admins as $admin) {
        $roleSlug = $admin['role'] === 'super_admin' ? 'super_admin' : 'admin';
        
        // Get role ID
        $stmt = $pdo->prepare("SELECT id FROM roles WHERE slug = :slug");
        $stmt->execute([':slug' => $roleSlug]);
        $role = $stmt->fetch();
        
        if ($role) {
            // Check if already assigned (with or without user_type based on column existence)
            if ($hasUserType) {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_roles 
                    WHERE user_id = :user_id AND role_id = :role_id AND user_type = 'admin'
                ");
                $stmt->execute([
                    ':user_id' => $admin['user_id'],
                    ':role_id' => $role['id']
                ]);
            } else {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_roles 
                    WHERE user_id = :user_id AND role_id = :role_id
                ");
                $stmt->execute([
                    ':user_id' => $admin['user_id'],
                    ':role_id' => $role['id']
                ]);
            }
            
            if (!$stmt->fetch()) {
                // Assign role
                if ($hasUserType) {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_roles (user_id, role_id, user_type, assigned_by)
                        VALUES (:user_id, :role_id, 'admin', :assigned_by)
                    ");
                    $stmt->execute([
                        ':user_id' => $admin['user_id'],
                        ':role_id' => $role['id'],
                        ':assigned_by' => $admin['user_id']
                    ]);
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_roles (user_id, role_id, assigned_by)
                        VALUES (:user_id, :role_id, :assigned_by)
                    ");
                    $stmt->execute([
                        ':user_id' => $admin['user_id'],
                        ':role_id' => $role['id'],
                        ':assigned_by' => $admin['user_id']
                    ]);
                }
                echo "✓ Assigned {$roleSlug} role to admin user {$admin['user_id']}\n";
            }
        }
    }
    
    // Assign default roles to existing organizers
    echo "\nAssigning default roles to existing organizers...\n";
    
    $stmt = $pdo->query("
        SELECT o.user_id 
        FROM organizers o
        INNER JOIN customers c ON o.user_id = c.id
        WHERE o.is_active = 1 AND c.is_active = 1
    ");
    $organizers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $stmt = $pdo->prepare("SELECT id FROM roles WHERE slug = 'organizer_manager'");
    $stmt->execute();
    $organizerRole = $stmt->fetch();
    
    if ($organizerRole) {
        foreach ($organizers as $organizer) {
            // Check if already assigned
            if ($hasUserType) {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_roles 
                    WHERE user_id = :user_id AND role_id = :role_id AND user_type = 'organizer'
                ");
                $stmt->execute([
                    ':user_id' => $organizer['user_id'],
                    ':role_id' => $organizerRole['id']
                ]);
            } else {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_roles 
                    WHERE user_id = :user_id AND role_id = :role_id
                ");
                $stmt->execute([
                    ':user_id' => $organizer['user_id'],
                    ':role_id' => $organizerRole['id']
                ]);
            }
            
            if (!$stmt->fetch()) {
                // Assign role
                if ($hasUserType) {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_roles (user_id, role_id, user_type, assigned_by)
                        VALUES (:user_id, :role_id, 'organizer', :assigned_by)
                    ");
                    $stmt->execute([
                        ':user_id' => $organizer['user_id'],
                        ':role_id' => $organizerRole['id'],
                        ':assigned_by' => $organizer['user_id']
                    ]);
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_roles (user_id, role_id, assigned_by)
                        VALUES (:user_id, :role_id, :assigned_by)
                    ");
                    $stmt->execute([
                        ':user_id' => $organizer['user_id'],
                        ':role_id' => $organizerRole['id'],
                        ':assigned_by' => $organizer['user_id']
                    ]);
                }
                echo "✓ Assigned organizer_manager role to organizer user {$organizer['user_id']}\n";
            }
        }
    }
    
    // Assign default roles to existing venues
    echo "\nAssigning default roles to existing venues...\n";
    
    $stmt = $pdo->query("
        SELECT v.user_id 
        FROM venues v
        INNER JOIN customers c ON v.user_id = c.id
        WHERE v.is_active = 1 AND c.is_active = 1
    ");
    $venues = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $stmt = $pdo->prepare("SELECT id FROM roles WHERE slug = 'venue_manager'");
    $stmt->execute();
    $venueRole = $stmt->fetch();
    
    if ($venueRole) {
        foreach ($venues as $venue) {
            // Check if already assigned
            if ($hasUserType) {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_roles 
                    WHERE user_id = :user_id AND role_id = :role_id AND user_type = 'venue'
                ");
                $stmt->execute([
                    ':user_id' => $venue['user_id'],
                    ':role_id' => $venueRole['id']
                ]);
            } else {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_roles 
                    WHERE user_id = :user_id AND role_id = :role_id
                ");
                $stmt->execute([
                    ':user_id' => $venue['user_id'],
                    ':role_id' => $venueRole['id']
                ]);
            }
            
            if (!$stmt->fetch()) {
                // Assign role
                if ($hasUserType) {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_roles (user_id, role_id, user_type, assigned_by)
                        VALUES (:user_id, :role_id, 'venue', :assigned_by)
                    ");
                    $stmt->execute([
                        ':user_id' => $venue['user_id'],
                        ':role_id' => $venueRole['id'],
                        ':assigned_by' => $venue['user_id']
                    ]);
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_roles (user_id, role_id, assigned_by)
                        VALUES (:user_id, :role_id, :assigned_by)
                    ");
                    $stmt->execute([
                        ':user_id' => $venue['user_id'],
                        ':role_id' => $venueRole['id'],
                        ':assigned_by' => $venue['user_id']
                    ]);
                }
                echo "✓ Assigned venue_manager role to venue user {$venue['user_id']}\n";
            }
        }
    }
    
    echo "\n✅ Migration completed successfully!\n";
    echo "\nYou can now:\n";
    echo "1. Access Roles & Permissions from admin panel\n";
    echo "2. Create custom roles\n";
    echo "3. Assign permissions to roles\n";
    echo "4. Assign roles to users\n\n";
    
} catch (Exception $e) {
    echo "\n❌ Migration failed: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
    exit(1);
}

