<?php
/**
 * Organizer - Sponsor Detail
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view sponsors
if (!hasPermission('organizer.sponsors.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view sponsors.'));
    exit;
}

$sponsorId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$sponsorId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/sponsors.php?error=' . urlencode('Sponsor ID is required.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility
try {
    $check = $pdo->query("SHOW COLUMNS FROM sponsors LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Verify ownership and fetch sponsor
$sql = "SELECT * FROM sponsors WHERE id = :id AND organizer_id = :organizer_id";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
$sponsor = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$sponsor) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/sponsors.php?error=' . urlencode('Sponsor not found or you do not have permission to view this sponsor.'));
    exit;
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($sponsor['name']); ?> | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'sponsors';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="page-header">
                <div>
                    <h1><?php echo htmlspecialchars($sponsor['name']); ?></h1>
                    <p>Sponsor Details</p>
                </div>
                <div>
                    <?php if (hasPermission('organizer.sponsors.edit')): ?>
                    <a href="<?php echo $baseUrl; ?>/organizer/pages/edit-sponsor.php?id=<?php echo $sponsor['id']; ?>" class="btn btn-primary">Edit Sponsor</a>
                    <?php endif; ?>
                    <a href="<?php echo $baseUrl; ?>/organizer/pages/sponsors.php" class="btn btn-secondary">Back to Sponsors</a>
                </div>
            </div>
            
            <div class="detail-container">
                <div class="detail-section">
                    <div class="detail-row">
                        <div class="detail-label">Logo:</div>
                        <div class="detail-value">
                            <?php if (!empty($sponsor['logo_image'])): ?>
                                <img src="<?php echo htmlspecialchars($sponsor['logo_image']); ?>" alt="<?php echo htmlspecialchars($sponsor['name']); ?>" style="max-width: 200px; border-radius: 8px;">
                            <?php else: ?>
                                <div style="width: 200px; height: 200px; background: #e5e7eb; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 60px;">🏷️</div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="detail-row">
                        <div class="detail-label">Name:</div>
                        <div class="detail-value"><strong><?php echo htmlspecialchars($sponsor['name']); ?></strong></div>
                    </div>
                    
                    <div class="detail-row">
                        <div class="detail-label">Status:</div>
                        <div class="detail-value">
                            <?php if ($sponsor['status'] === 'published'): ?>
                                <span class="badge badge-success">Published</span>
                            <?php else: ?>
                                <span class="badge badge-warning">Unpublished</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if (!empty($sponsor['description'])): ?>
                    <div class="detail-row">
                        <div class="detail-label">Description:</div>
                        <div class="detail-value"><?php echo nl2br(htmlspecialchars($sponsor['description'])); ?></div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($sponsor['website_url'])): ?>
                    <div class="detail-row">
                        <div class="detail-label">Website:</div>
                        <div class="detail-value">
                            <a href="<?php echo htmlspecialchars($sponsor['website_url']); ?>" target="_blank">
                                <?php echo htmlspecialchars($sponsor['website_url']); ?>
                            </a>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <div class="detail-row">
                        <div class="detail-label">Created:</div>
                        <div class="detail-value"><?php echo date('F j, Y g:i A', strtotime($sponsor['created_at'])); ?></div>
                    </div>
                    
                    <?php if ($sponsor['updated_at'] !== $sponsor['created_at']): ?>
                    <div class="detail-row">
                        <div class="detail-label">Last Updated:</div>
                        <div class="detail-value"><?php echo date('F j, Y g:i A', strtotime($sponsor['updated_at'])); ?></div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
    
    <style>
        .detail-container {
            background: white;
            border-radius: 8px;
            padding: 2rem;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .detail-section {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
        }
        .detail-row {
            display: grid;
            grid-template-columns: 150px 1fr;
            gap: 1rem;
            align-items: start;
        }
        .detail-label {
            font-weight: 600;
            color: #374151;
        }
        .detail-value {
            color: #111827;
        }
        @media (max-width: 768px) {
            .detail-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</body>
</html>


