<?php
/**
 * Organizer - Sales & Finance
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view sales
if (!hasPermission('organizer.sales.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view sales.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Get sales statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT b.id) as total_bookings,
        SUM(b.final_amount) as total_revenue,
        COUNT(DISTINCT t.id) as total_tickets_sold,
        AVG(b.final_amount) as avg_booking_value
    FROM bookings b
    INNER JOIN events e ON b.event_id = e.id
    LEFT JOIN tickets t ON b.id = t.booking_id
    WHERE e.organizer_id = :organizer_id AND b.payment_status = 'completed'
");
$stmt->execute([':organizer_id' => $organizer['id']]);
$sales_stats = $stmt->fetch();

// Get recent bookings
$stmt = $pdo->prepare("
    SELECT 
        b.*,
        e.name as event_name,
        c.full_name as customer_name,
        c.email as customer_email,
        COUNT(t.id) as ticket_count
    FROM bookings b
    INNER JOIN events e ON b.event_id = e.id
    INNER JOIN customers c ON b.customer_id = c.id
    LEFT JOIN tickets t ON b.id = t.booking_id
    WHERE e.organizer_id = :organizer_id
    GROUP BY b.id
    ORDER BY b.created_at DESC
    LIMIT 50
");
$stmt->execute([':organizer_id' => $organizer['id']]);
$bookings = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sales & Finance | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'sales';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Sales & Finance</h1>
                
                <!-- Statistics -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">💰</div>
                        <div class="stat-info">
                            <div class="stat-value">$<?php echo number_format($sales_stats['total_revenue'] ?? 0, 2); ?></div>
                            <div class="stat-label">Total Revenue</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">🎟️</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($sales_stats['total_tickets_sold'] ?? 0); ?></div>
                            <div class="stat-label">Tickets Sold</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">📦</div>
                        <div class="stat-info">
                            <div class="stat-value"><?php echo number_format($sales_stats['total_bookings'] ?? 0); ?></div>
                            <div class="stat-label">Total Bookings</div>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">📊</div>
                        <div class="stat-info">
                            <div class="stat-value">$<?php echo number_format($sales_stats['avg_booking_value'] ?? 0, 2); ?></div>
                            <div class="stat-label">Avg Booking Value</div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Bookings -->
                <div class="section">
                    <h2>Recent Bookings</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Booking Ref</th>
                                    <th>Event</th>
                                    <th>Customer</th>
                                    <th>Tickets</th>
                                    <th>Amount</th>
                                    <th>Payment Status</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($bookings)): ?>
                                <tr>
                                    <td colspan="8" style="text-align: center; padding: 2rem;">
                                        No bookings yet.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($bookings as $booking): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($booking['booking_reference']); ?></td>
                                        <td><?php echo htmlspecialchars($booking['event_name']); ?></td>
                                        <td><?php echo htmlspecialchars($booking['customer_name'] ?? $booking['customer_email']); ?></td>
                                        <td><?php echo $booking['ticket_count']; ?></td>
                                        <td>$<?php echo number_format($booking['final_amount'], 2); ?></td>
                                        <td><span class="status-badge status-<?php echo $booking['payment_status']; ?>"><?php echo ucfirst($booking['payment_status']); ?></span></td>
                                        <td><?php echo date('M d, Y H:i', strtotime($booking['booking_date'])); ?></td>
                                        <td>
                                            <a href="booking-detail.php?id=<?php echo $booking['id']; ?>" class="btn-sm btn-primary">View</a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
</body>
</html>

