<?php
/**
 * Organizer - Refund Management
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view refunds
if (!hasPermission('organizer.refunds.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view refunds.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Get refund requests
$stmt = $pdo->prepare("
    SELECT 
        r.*,
        b.booking_reference,
        e.name as event_name,
        c.full_name as customer_name,
        c.email as customer_email
    FROM refunds r
    INNER JOIN bookings b ON r.booking_id = b.id
    INNER JOIN events e ON b.event_id = e.id
    INNER JOIN customers c ON r.requested_by = c.id
    WHERE e.organizer_id = :organizer_id
    ORDER BY r.created_at DESC
    LIMIT 50
");
$stmt->execute([':organizer_id' => $organizer['id']]);
$refunds = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Refund Management | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'refunds';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Refund Management</h1>
                
                <!-- Refund Requests -->
                <div class="section">
                    <h2>Refund Requests</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Booking Ref</th>
                                    <th>Event</th>
                                    <th>Customer</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Requested</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($refunds)): ?>
                                <tr>
                                    <td colspan="9" style="text-align: center; padding: 2rem;">
                                        No refund requests found.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($refunds as $refund): ?>
                                    <tr>
                                        <td><?php echo $refund['id']; ?></td>
                                        <td><?php echo htmlspecialchars($refund['booking_reference']); ?></td>
                                        <td><?php echo htmlspecialchars($refund['event_name']); ?></td>
                                        <td><?php echo htmlspecialchars($refund['customer_name'] ?? $refund['customer_email']); ?></td>
                                        <td><?php echo ucfirst($refund['request_type']); ?></td>
                                        <td>$<?php echo number_format($refund['amount'], 2); ?></td>
                                        <td><span class="status-badge status-<?php echo $refund['status']; ?>"><?php echo ucfirst($refund['status']); ?></span></td>
                                        <td><?php echo date('M d, Y', strtotime($refund['created_at'])); ?></td>
                                        <td>
                                            <?php if ($refund['status'] === 'pending'): ?>
                                                <a href="<?php echo $baseUrl; ?>/api/organizer/approverefundapi.php?id=<?php echo $refund['id']; ?>" class="btn-sm btn-primary" onclick="return confirm('Approve this refund?')">Approve</a>
                                                <a href="<?php echo $baseUrl; ?>/api/organizer/rejectrefundapi.php?id=<?php echo $refund['id']; ?>" class="btn-sm btn-secondary" onclick="return confirm('Reject this refund?')">Reject</a>
                                            <?php else: ?>
                                                <a href="refund-detail.php?id=<?php echo $refund['id']; ?>" class="btn-sm btn-primary">View</a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
</body>
</html>

