<?php
/**
 * Organizer - Promotions
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view promotions
if (!hasPermission('organizer.promotions.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view promotions.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Get organizer's events
$stmt = $pdo->prepare("SELECT id, name FROM events WHERE organizer_id = :organizer_id ORDER BY name ASC");
$stmt->execute([':organizer_id' => $organizer['id']]);
$events = $stmt->fetchAll();

// Get organizer's coupons
$stmt = $pdo->prepare("
    SELECT c.*, COUNT(cu.id) as usage_count
    FROM coupons c
    LEFT JOIN coupon_usage cu ON c.id = cu.coupon_id
    WHERE c.created_by_type = 'organizer' AND c.created_by = :organizer_id
    GROUP BY c.id
    ORDER BY c.created_at DESC
");
$stmt->execute([':organizer_id' => $organizer['id']]);
$coupons = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Promotions | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'promotions';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Promotions & Marketing</h1>
                
                <!-- Create Coupon -->
                <div class="section">
                    <h2>Create Promo Code</h2>
                    <form method="POST" action="<?php echo $baseUrl; ?>/api/organizer/createcouponapi.php">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="code">Coupon Code *</label>
                                <input type="text" id="code" name="code" required pattern="[A-Z0-9]+" placeholder="SAVE20">
                            </div>
                            
                            <div class="form-group">
                                <label for="name">Coupon Name</label>
                                <input type="text" id="name" name="name" placeholder="Summer Sale">
                            </div>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="discount_type">Discount Type *</label>
                                <select id="discount_type" name="discount_type" required>
                                    <option value="percentage">Percentage (%)</option>
                                    <option value="fixed">Fixed Amount ($)</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="discount_value">Discount Value *</label>
                                <input type="number" id="discount_value" name="discount_value" step="0.01" required>
                            </div>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="valid_from">Valid From *</label>
                                <input type="datetime-local" id="valid_from" name="valid_from" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="valid_until">Valid Until *</label>
                                <input type="datetime-local" id="valid_until" name="valid_until" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="applicable_to">Applicable To</label>
                            <select id="applicable_to" name="applicable_to">
                                <option value="all">All Events</option>
                                <option value="event">Specific Event</option>
                            </select>
                        </div>
                        
                        <div class="form-group" id="eventSelectGroup" style="display: none;">
                            <label for="applicable_id">Select Event</label>
                            <select id="applicable_id" name="applicable_id">
                                <?php foreach ($events as $event): ?>
                                    <option value="<?php echo $event['id']; ?>"><?php echo htmlspecialchars($event['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <button type="submit" class="btn-sm btn-primary">Create Coupon</button>
                    </form>
                </div>
                
                <!-- Existing Coupons -->
                <div class="section">
                    <h2>My Promo Codes</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Code</th>
                                    <th>Name</th>
                                    <th>Discount</th>
                                    <th>Valid From</th>
                                    <th>Valid Until</th>
                                    <th>Usage</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($coupons)): ?>
                                <tr>
                                    <td colspan="8" style="text-align: center; padding: 2rem;">
                                        No promo codes created yet. Create your first promo code above.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($coupons as $coupon): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($coupon['code']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($coupon['name'] ?? 'N/A'); ?></td>
                                        <td>
                                            <?php if ($coupon['discount_type'] === 'percentage'): ?>
                                                <?php echo $coupon['discount_value']; ?>%
                                            <?php else: ?>
                                                $<?php echo number_format($coupon['discount_value'], 2); ?>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('M d, Y', strtotime($coupon['valid_from'])); ?></td>
                                        <td><?php echo date('M d, Y', strtotime($coupon['valid_until'])); ?></td>
                                        <td><?php echo $coupon['usage_count']; ?> / <?php echo $coupon['usage_limit'] ?? '∞'; ?></td>
                                        <td>
                                            <?php if ($coupon['is_active'] && strtotime($coupon['valid_until']) > time()): ?>
                                                <span class="status-badge status-approved">Active</span>
                                            <?php else: ?>
                                                <span class="status-badge status-rejected">Inactive</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="#" class="btn-sm btn-secondary">Edit</a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    document.getElementById('applicable_to').addEventListener('change', function() {
        const eventSelect = document.getElementById('eventSelectGroup');
        if (this.value === 'event') {
            eventSelect.style.display = 'block';
        } else {
            eventSelect.style.display = 'none';
        }
    });
    </script>
</body>
</html>

