<?php
/**
 * Organizer - My Events
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view events
if (!hasPermission('organizer.events.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view events.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Helper: Get artists mapped to an event for this organizer
function getEventArtistsForRow(PDO $pdo, int $eventId, int $organizerId): array {
    try {
        // Ensure pivot table exists
        $pdo->query("SELECT 1 FROM event_artists LIMIT 1");
    } catch (Exception $e) {
        // Mapping table not present yet
        return [];
    }

    try {
        $stmt = $pdo->prepare("
            SELECT a.id, a.name 
            FROM artists a
            INNER JOIN event_artists ea ON a.id = ea.artist_id
            WHERE ea.event_id = :event_id
              AND a.organizer_id = :organizer_id
              AND (a.deleted_at IS NULL OR a.deleted_at IS NULL)
            ORDER BY a.name ASC
        ");
        $stmt->execute([
            ':event_id' => $eventId,
            ':organizer_id' => $organizerId
        ]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Exception $e) {
        error_log('Error fetching event artists: ' . $e->getMessage());
        return [];
    }
}

// Helper: Get sponsors mapped to an event for this organizer
function getEventSponsorsForRow(PDO $pdo, int $eventId, int $organizerId): array {
    try {
        $pdo->query("SELECT 1 FROM event_sponsors LIMIT 1");
    } catch (Exception $e) {
        return [];
    }

    try {
        $stmt = $pdo->prepare("
            SELECT s.id, s.name 
            FROM sponsors s
            INNER JOIN event_sponsors es ON s.id = es.sponsor_id
            WHERE es.event_id = :event_id
              AND s.organizer_id = :organizer_id
              AND s.deleted_at IS NULL
            ORDER BY s.name ASC
        ");
        $stmt->execute([
            ':event_id' => $eventId,
            ':organizer_id' => $organizerId
        ]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Exception $e) {
        error_log('Error fetching event sponsors: ' . $e->getMessage());
        return [];
    }
}

// Soft-delete backward compatibility: deleted_at may not exist if migration wasn't run
try {
    $check = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Get status filter from URL
$statusFilter = $_GET['status'] ?? 'all';

// Get events (exclude soft-deleted if supported)
$sql = "
    SELECT * FROM events
    WHERE organizer_id = :organizer_id
";
$params = [':organizer_id' => $organizer['id']];

if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

// Apply status filter
if ($statusFilter !== 'all') {
    if ($statusFilter === 'published') {
        $sql .= " AND status IN ('published', 'approved', 'live')";
    } else {
        $sql .= " AND status = :status";
        $params[':status'] = $statusFilter;
    }
}

$sql .= " ORDER BY created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$events = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Events | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <?php 
    require_once __DIR__ . '/../../includes/url_helper.php';
    $baseUrl = getCorrectBaseUrl();
    // getCorrectBaseUrl() already returns full URL, no need for fallback
    ?>
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'events';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                    <h1>My Events</h1>
                    <?php if (hasPermission('organizer.events.create')): ?>
                    <a href="create-event.php" class="btn-sm btn-primary">+ Create Event</a>
                    <?php endif; ?>
                </div>
                
                <div class="section">
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Event Name</th>
                                    <th>Artists</th>
                                    <th>Sponsors</th>
                                    <th>Status</th>
                                    <th>Start Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($events)): ?>
                                <tr>
                                    <td colspan="4" style="text-align: center; padding: 2rem; color: #6b7280;">
                                        No events found. <a href="create-event.php">Create your first event</a>
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($events as $event): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($event['name']); ?></td>
                                        <td>
                                            <?php 
                                            $eventArtists = getEventArtistsForRow($pdo, (int)$event['id'], (int)$organizer['id']);
                                            if (empty($eventArtists)): ?>
                                                <span style="color: #9ca3af;">No artists</span>
                                            <?php else: 
                                                $names = array_map(function($a) { return $a['name']; }, $eventArtists);
                                            ?>
                                                <span><?php echo htmlspecialchars(implode(', ', $names)); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php 
                                            $eventSponsors = getEventSponsorsForRow($pdo, (int)$event['id'], (int)$organizer['id']);
                                            if (empty($eventSponsors)): ?>
                                                <span style="color: #9ca3af;">No sponsors</span>
                                            <?php else: 
                                                $sNames = array_map(function($s) { return $s['name']; }, $eventSponsors);
                                            ?>
                                                <span><?php echo htmlspecialchars(implode(', ', $sNames)); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td><span class="status-badge status-<?php echo $event['status']; ?>"><?php echo ucfirst($event['status']); ?></span></td>
                                        <td><?php echo date('M d, Y', strtotime($event['start_date'])); ?></td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem;">
                                                <a href="event-detail.php?id=<?php echo $event['id']; ?>" class="btn-sm btn-primary">View</a>
                                                <?php if (hasPermission('organizer.events.edit')): ?>
                                                <a href="edit-event.php?id=<?php echo $event['id']; ?>" class="btn-sm btn-secondary">Edit</a>
                                                <?php endif; ?>
                                                <?php if (hasPermission('organizer.events.delete')): ?>
                                                <button onclick="deleteEvent(<?php echo $event['id']; ?>, '<?php echo htmlspecialchars(addslashes($event['name'])); ?>')" class="btn-sm" style="background: #dc2626; color: white; border: none; cursor: pointer;">Delete</button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    function deleteEvent(eventId, eventName) {
        if (!confirm('Are you sure you want to delete "' + eventName + '"? This action cannot be undone.')) {
            return;
        }
        
        fetch('<?php echo $baseUrl; ?>/api/organizer/createeventapi.php?action=delete&id=' + eventId, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Event deleted successfully');
                location.reload();
            } else {
                alert('Error: ' + (data.message || 'Failed to delete event'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while deleting the event');
        });
    }
    </script>
</body>
</html>

