<?php
/**
 * Organizer - Event Detail
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view events
if (!hasPermission('organizer.events.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view events.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility: deleted_at may not exist if migration wasn't run
try {
    $check = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Get event ID
$eventId = $_GET['id'] ?? null;
if (!$eventId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/events.php?error=' . urlencode('Event ID is required'));
    exit;
}

// Get event and verify ownership (exclude soft-deleted only if column exists)
$sql = "
    SELECT e.*, v.venue_name, o.company_name as organizer_name
    FROM events e
    LEFT JOIN venues v ON e.venue_id = v.id
    LEFT JOIN organizers o ON e.organizer_id = o.id
    WHERE e.id = :id AND e.organizer_id = :organizer_id
";
if ($hasDeletedAt) {
    $sql .= " AND e.deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $eventId, ':organizer_id' => $organizer['id']]);
$event = $stmt->fetch();

if (!$event) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/events.php?error=' . urlencode('Event not found'));
    exit;
}

// Get event artists
try {
    $artistsStmt = $pdo->prepare("
        SELECT a.* FROM artists a
        INNER JOIN event_artists ea ON a.id = ea.artist_id
        WHERE ea.event_id = :event_id
        AND a.organizer_id = :organizer_id
        AND a.deleted_at IS NULL
        ORDER BY a.name ASC
    ");
    $artistsStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
    $eventArtists = $artistsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Decode social links
    foreach ($eventArtists as &$artist) {
        if (!empty($artist['social_links'])) {
            $artist['social_links'] = json_decode($artist['social_links'], true) ?: [];
        }
    }
} catch (Exception $e) {
    $eventArtists = [];
}

// Get event sponsors
try {
    $sponsorsStmt = $pdo->prepare("
        SELECT s.* FROM sponsors s
        INNER JOIN event_sponsors es ON s.id = es.sponsor_id
        WHERE es.event_id = :event_id
        AND s.organizer_id = :organizer_id
        AND s.deleted_at IS NULL
        ORDER BY s.name ASC
    ");
    $sponsorsStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
    $eventSponsors = $sponsorsStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $eventSponsors = [];
}

// Get event agendas
try {
    $agendasStmt = $pdo->prepare("
        SELECT a.* FROM agendas a
        INNER JOIN event_agendas ea ON a.id = ea.agenda_id
        WHERE ea.event_id = :event_id
        AND a.organizer_id = :organizer_id
        AND a.deleted_at IS NULL
        ORDER BY COALESCE(a.start_time, a.created_at) ASC
    ");
    $agendasStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
    $eventAgendas = $agendasStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $eventAgendas = [];
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Event Details | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'events';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                    <h1><?php echo htmlspecialchars($event['name']); ?></h1>
                    <div style="display: flex; gap: 0.5rem;">
                        <?php if (hasPermission('organizer.events.edit')): ?>
                        <a href="edit-event.php?id=<?php echo $event['id']; ?>" class="btn-sm btn-secondary">Edit</a>
                        <?php endif; ?>
                        <a href="events.php" class="btn-sm btn-secondary">Back to Events</a>
                    </div>
                </div>
                
                <!-- Event Status -->
                <div class="section">
                    <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                        <strong>Status:</strong>
                        <span class="status-badge status-<?php echo $event['status']; ?>"><?php echo ucfirst($event['status']); ?></span>
                    </div>
                </div>
                
                <!-- Event Information -->
                <div class="section">
                    <h2>Event Information</h2>
                    
                    <?php if ($event['banner_image']): ?>
                    <div style="margin-bottom: 1.5rem;">
                        <img src="<?php echo htmlspecialchars($event['banner_image']); ?>" alt="<?php echo htmlspecialchars($event['name']); ?>" style="max-width: 100%; height: auto; border-radius: 8px;">
                    </div>
                    <?php endif; ?>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1.5rem;">
                        <div>
                            <h3 style="margin-bottom: 0.5rem;">Basic Details</h3>
                            <p><strong>Event Name:</strong> <?php echo htmlspecialchars($event['name']); ?></p>
                            <?php if ($event['short_description']): ?>
                            <p><strong>Short Description:</strong> <?php echo htmlspecialchars($event['short_description']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['description']): ?>
                            <p><strong>Description:</strong><br><?php echo nl2br(htmlspecialchars($event['description'])); ?></p>
                            <?php endif; ?>
                            <?php if ($event['event_type']): ?>
                            <p><strong>Event Type:</strong> <?php echo ucfirst($event['event_type']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['category']): ?>
                            <p><strong>Category:</strong> <?php echo ucfirst($event['category']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['genre']): ?>
                            <p><strong>Genre:</strong> <?php echo htmlspecialchars($event['genre']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['language']): ?>
                            <p><strong>Language:</strong> <?php echo htmlspecialchars($event['language']); ?></p>
                            <?php endif; ?>
                        </div>
                        
                        <div>
                            <h3 style="margin-bottom: 0.5rem;">Date & Location</h3>
                            <p><strong>Start Date:</strong> <?php echo date('F d, Y h:i A', strtotime($event['start_date'])); ?></p>
                            <p><strong>End Date:</strong> <?php echo date('F d, Y h:i A', strtotime($event['end_date'])); ?></p>
                            <?php if ($event['venue_name']): ?>
                            <p><strong>Venue:</strong> <?php echo htmlspecialchars($event['venue_name']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['address']): ?>
                            <p><strong>Address:</strong> <?php echo htmlspecialchars($event['address']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['city']): ?>
                            <p><strong>City:</strong> <?php echo htmlspecialchars($event['city']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['state']): ?>
                            <p><strong>State:</strong> <?php echo htmlspecialchars($event['state']); ?></p>
                            <?php endif; ?>
                            <?php if ($event['country']): ?>
                            <p><strong>Country:</strong> <?php echo htmlspecialchars($event['country']); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Additional Information -->
                <?php if ($event['attendees_range'] || $event['pricing'] || $event['age_restriction'] || $event['website_url']): ?>
                <div class="section">
                    <h2>Additional Information</h2>
                    <?php if ($event['attendees_range']): ?>
                    <p><strong>Expected Attendees:</strong> <?php echo htmlspecialchars($event['attendees_range']); ?></p>
                    <?php endif; ?>
                    <?php if ($event['pricing']): ?>
                    <p><strong>Pricing:</strong> <?php echo htmlspecialchars($event['pricing']); ?></p>
                    <?php endif; ?>
                    <?php if ($event['age_restriction']): ?>
                    <p><strong>Age Restriction:</strong> <?php echo htmlspecialchars($event['age_restriction']); ?></p>
                    <?php endif; ?>
                    <?php if ($event['website_url']): ?>
                    <p><strong>Website:</strong> <a href="<?php echo htmlspecialchars($event['website_url']); ?>" target="_blank"><?php echo htmlspecialchars($event['website_url']); ?></a></p>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <!-- Policies -->
                <?php if ($event['refund_policy']): ?>
                <div class="section">
                    <h2>Refund Policy</h2>
                    <p><?php echo nl2br(htmlspecialchars($event['refund_policy'])); ?></p>
                </div>
                <?php endif; ?>
                
                <!-- Event Artists -->
                <?php if (!empty($eventArtists)): ?>
                <div class="section">
                    <h2>Artists</h2>
                    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 1.5rem; margin-top: 1rem;">
                        <?php foreach ($eventArtists as $artist): ?>
                        <div style="border: 1px solid #e5e7eb; border-radius: 8px; padding: 1rem; text-align: center;">
                            <?php if ($artist['profile_image']): ?>
                                <img src="<?php echo htmlspecialchars($artist['profile_image']); ?>" alt="<?php echo htmlspecialchars($artist['name']); ?>" style="width: 100px; height: 100px; object-fit: cover; border-radius: 50%; margin-bottom: 0.5rem;">
                            <?php else: ?>
                                <div style="width: 100px; height: 100px; background: #e5e7eb; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 0.5rem; font-size: 40px;">🎤</div>
                            <?php endif; ?>
                            <h3 style="margin: 0.5rem 0; font-size: 1rem;"><?php echo htmlspecialchars($artist['name']); ?></h3>
                            <?php if ($artist['bio']): ?>
                                <p style="font-size: 0.875rem; color: #6b7280; margin: 0.5rem 0;">
                                    <?php echo htmlspecialchars(mb_substr($artist['bio'], 0, 100)) . (mb_strlen($artist['bio']) > 100 ? '...' : ''); ?>
                                </p>
                            <?php endif; ?>
                            <a href="<?php echo $baseUrl; ?>/organizer/pages/artist-detail.php?id=<?php echo $artist['id']; ?>" class="btn btn-sm btn-secondary" style="margin-top: 0.5rem;">View Artist</a>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Event Sponsors -->
                <?php if (!empty($eventSponsors)): ?>
                <div class="section">
                    <h2>Sponsors</h2>
                    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 1.5rem; margin-top: 1rem;">
                        <?php foreach ($eventSponsors as $sponsor): ?>
                        <div style="border: 1px solid #e5e7eb; border-radius: 8px; padding: 1rem; text-align: center;">
                            <?php if (!empty($sponsor['logo_image'])): ?>
                                <img src="<?php echo htmlspecialchars($sponsor['logo_image']); ?>" alt="<?php echo htmlspecialchars($sponsor['name']); ?>" style="width: 100px; height: 100px; object-fit: cover; border-radius: 12px; margin-bottom: 0.5rem;">
                            <?php else: ?>
                                <div style="width: 100px; height: 100px; background: #e5e7eb; border-radius: 12px; display: flex; align-items: center; justify-content: center; margin: 0 auto 0.5rem; font-size: 40px;">🏷️</div>
                            <?php endif; ?>
                            <h3 style="margin: 0.5rem 0; font-size: 1rem;"><?php echo htmlspecialchars($sponsor['name']); ?></h3>
                            <?php if (!empty($sponsor['description'])): ?>
                                <p style="font-size: 0.875rem; color: #6b7280; margin: 0.5rem 0;">
                                    <?php echo htmlspecialchars(mb_substr($sponsor['description'], 0, 100)) . (mb_strlen($sponsor['description']) > 100 ? '...' : ''); ?>
                                </p>
                            <?php endif; ?>
                            <?php if (!empty($sponsor['website_url'])): ?>
                                <p style="margin-top: 0.5rem;">
                                    <a href="<?php echo htmlspecialchars($sponsor['website_url']); ?>" target="_blank">Visit Website</a>
                                </p>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Event Agenda -->
                <?php if (!empty($eventAgendas)): ?>
                <div class="section">
                    <h2>Agenda</h2>
                    <div style="display: flex; flex-direction: column; gap: 0.75rem; margin-top: 0.5rem;">
                        <?php foreach ($eventAgendas as $agenda): ?>
                        <div style="border: 1px solid #e5e7eb; border-radius: 8px; padding: 0.75rem 1rem; display: flex; flex-direction: column; gap: 0.25rem;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <strong><?php echo htmlspecialchars($agenda['title']); ?></strong>
                                <?php if ($agenda['start_time'] || $agenda['end_time']): ?>
                                    <?php
                                        $start = $agenda['start_time'] ? date('M d, Y g:i A', strtotime($agenda['start_time'])) : '';
                                        $end = $agenda['end_time'] ? date('M d, Y g:i A', strtotime($agenda['end_time'])) : '';
                                    ?>
                                    <span style="font-size: 0.8125rem; color: #6b7280;">
                                        <?php echo htmlspecialchars(trim($start . ($end ? ' - ' . $end : ''))); ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                            <?php if (!empty($agenda['description'])): ?>
                            <div style="font-size: 0.875rem; color: #4b5563;">
                                <?php echo nl2br(htmlspecialchars($agenda['description'])); ?>
                            </div>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Created/Updated Info -->
                <div class="section">
                    <p style="color: #6b7280; font-size: 0.875rem;">
                        <strong>Created:</strong> <?php echo date('F d, Y h:i A', strtotime($event['created_at'])); ?><br>
                        <?php if ($event['updated_at'] && $event['updated_at'] !== $event['created_at']): ?>
                        <strong>Last Updated:</strong> <?php echo date('F d, Y h:i A', strtotime($event['updated_at'])); ?>
                        <?php endif; ?>
                    </p>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
</body>
</html>

