<?php
/**
 * Organizer - Edit Sponsor
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to edit sponsors
if (!hasPermission('organizer.sponsors.edit')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to edit sponsors.'));
    exit;
}

$sponsorId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$sponsorId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/sponsors.php?error=' . urlencode('Sponsor ID is required.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility
try {
    $check = $pdo->query("SHOW COLUMNS FROM sponsors LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Verify ownership and fetch sponsor
$sql = "SELECT * FROM sponsors WHERE id = :id AND organizer_id = :organizer_id";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
$sponsor = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$sponsor) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/sponsors.php?error=' . urlencode('Sponsor not found or you do not have permission to edit this sponsor.'));
    exit;
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Sponsor | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'sponsors';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Edit Sponsor</h1>
                
                <div class="section">
                    <form id="editSponsorForm" method="POST" action="<?php echo $baseUrl; ?>/api/organizer/sponsorsapi.php?action=update&id=<?php echo $sponsor['id']; ?>">
                        <div class="form-group">
                            <label for="name">Sponsor Name *</label>
                            <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($sponsor['name']); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea id="description" name="description" rows="4" placeholder="Describe this sponsor..."><?php echo htmlspecialchars($sponsor['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="logo_image">Logo Image URL</label>
                            <input type="url" id="logo_image" name="logo_image" value="<?php echo htmlspecialchars($sponsor['logo_image'] ?? ''); ?>" placeholder="https://example.com/logo.jpg">
                            <small class="form-text">Enter a URL to the sponsor's logo image</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="website_url">Website URL</label>
                            <input type="url" id="website_url" name="website_url" value="<?php echo htmlspecialchars($sponsor['website_url'] ?? ''); ?>" placeholder="https://example.com">
                        </div>
                        
                        <div class="form-group">
                            <label for="status">Status</label>
                            <select id="status" name="status">
                                <option value="unpublished" <?php echo ($sponsor['status'] === 'unpublished') ? 'selected' : ''; ?>>Unpublished</option>
                                <option value="published" <?php echo ($sponsor['status'] === 'published') ? 'selected' : ''; ?>>Published</option>
                            </select>
                            <small class="form-text">Unpublished sponsors are not visible publicly</small>
                        </div>
                        
                        <div style="display: flex; gap: 1rem;">
                            <button type="submit" class="btn-sm btn-primary">Update Sponsor</button>
                            <a href="sponsors.php" class="btn-sm btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        const BASE_URL = '<?php echo $baseUrl; ?>';
    </script>
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    document.getElementById('editSponsorForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = {};
        
        for (let [key, value] of formData.entries()) {
            if (value.trim()) {
                data[key] = value.trim();
            }
        }
        
        try {
            const response = await fetch(this.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'Failed to update sponsor');
            }
            
            alert('Sponsor updated successfully');
            window.location.href = BASE_URL + '/organizer/pages/sponsors.php';
            
        } catch (error) {
            console.error('Error updating sponsor:', error);
            alert('Error: ' + error.message);
        }
    });
    </script>
</body>
</html>


