<?php
/**
 * Organizer - Edit Event
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to edit events
if (!hasPermission('organizer.events.edit')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to edit events. Please contact your administrator.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility: deleted_at may not exist if migration wasn't run
try {
    $check = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Get event ID
$eventId = $_GET['id'] ?? null;
if (!$eventId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/events.php?error=' . urlencode('Event ID is required'));
    exit;
}

// Get event and verify ownership (exclude soft-deleted only if column exists)
$sql = "
    SELECT * FROM events 
    WHERE id = :id AND organizer_id = :organizer_id
";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $eventId, ':organizer_id' => $organizer['id']]);
$event = $stmt->fetch();

if (!$event) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/events.php?error=' . urlencode('Event not found or you do not have permission to edit this event.'));
    exit;
}

// Get categories
$stmt = $pdo->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY name ASC");
$categories = $stmt->fetchAll();

// Get venues
$stmt = $pdo->query("SELECT * FROM venues WHERE is_active = 1 ORDER BY venue_name ASC");
$venues = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Event | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'events';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Edit Event</h1>
                
                <div class="section">
                    <form id="editEventForm" method="POST" action="<?php echo $baseUrl; ?>/api/organizer/createeventapi.php?action=update&id=<?php echo $event['id']; ?>">
                        <div class="form-group">
                            <label for="name">Event Name *</label>
                            <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($event['name']); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="short_description">Short Description</label>
                            <input type="text" id="short_description" name="short_description" value="<?php echo htmlspecialchars($event['short_description'] ?? ''); ?>" maxlength="500">
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Full Description *</label>
                            <textarea id="description" name="description" rows="6" required><?php echo htmlspecialchars($event['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="event_type">Event Type</label>
                                <select id="event_type" name="event_type">
                                    <option value="">Select Type</option>
                                    <option value="conference" <?php echo ($event['event_type'] === 'conference') ? 'selected' : ''; ?>>Conference</option>
                                    <option value="exhibition" <?php echo ($event['event_type'] === 'exhibition') ? 'selected' : ''; ?>>Exhibition</option>
                                    <option value="festival" <?php echo ($event['event_type'] === 'festival') ? 'selected' : ''; ?>>Festival</option>
                                    <option value="concert" <?php echo ($event['event_type'] === 'concert') ? 'selected' : ''; ?>>Concert</option>
                                    <option value="workshop" <?php echo ($event['event_type'] === 'workshop') ? 'selected' : ''; ?>>Workshop</option>
                                    <option value="summit" <?php echo ($event['event_type'] === 'summit') ? 'selected' : ''; ?>>Summit</option>
                                    <option value="trade-show" <?php echo ($event['event_type'] === 'trade-show') ? 'selected' : ''; ?>>Trade Show</option>
                                    <option value="sports" <?php echo ($event['event_type'] === 'sports') ? 'selected' : ''; ?>>Sports</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="category">Category</label>
                                <select id="category" name="category">
                                    <option value="">Select Category</option>
                                    <?php foreach ($categories as $cat): ?>
                                        <option value="<?php echo htmlspecialchars($cat['slug']); ?>" <?php echo ($event['category'] === $cat['slug']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($cat['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="start_date">Start Date & Time *</label>
                                <input type="datetime-local" id="start_date" name="start_date" value="<?php echo date('Y-m-d\TH:i', strtotime($event['start_date'])); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="end_date">End Date & Time *</label>
                                <input type="datetime-local" id="end_date" name="end_date" value="<?php echo date('Y-m-d\TH:i', strtotime($event['end_date'])); ?>" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="venue_id">Venue</label>
                            <select id="venue_id" name="venue_id">
                                <option value="">Select Venue (Optional)</option>
                                <?php foreach ($venues as $venue): ?>
                                    <option value="<?php echo $venue['id']; ?>" <?php echo ($event['venue_id'] == $venue['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($venue['venue_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="city">City</label>
                                <input type="text" id="city" name="city" value="<?php echo htmlspecialchars($event['city'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="state">State</label>
                                <input type="text" id="state" name="state" value="<?php echo htmlspecialchars($event['state'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="country">Country *</label>
                                <input type="text" id="country" name="country" value="<?php echo htmlspecialchars($event['country'] ?? ''); ?>" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="address">Address</label>
                            <textarea id="address" name="address" rows="2"><?php echo htmlspecialchars($event['address'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="banner_image">Banner Image URL</label>
                            <input type="url" id="banner_image" name="banner_image" value="<?php echo htmlspecialchars($event['banner_image'] ?? ''); ?>" placeholder="https://example.com/image.jpg">
                        </div>
                        
                        <div class="form-group">
                            <label for="website_url">Event Website URL</label>
                            <input type="url" id="website_url" name="website_url" value="<?php echo htmlspecialchars($event['website_url'] ?? ''); ?>" placeholder="https://example.com">
                        </div>
                        
                        <div class="form-group">
                            <label for="attendees_range">Expected Attendees</label>
                            <input type="text" id="attendees_range" name="attendees_range" value="<?php echo htmlspecialchars($event['attendees_range'] ?? ''); ?>" placeholder="e.g., 1000+, 500-1000">
                        </div>
                        
                        <div class="form-group">
                            <label for="pricing">Pricing</label>
                            <input type="text" id="pricing" name="pricing" value="<?php echo htmlspecialchars($event['pricing'] ?? ''); ?>" placeholder="e.g., $50-$200">
                        </div>
                        
                        <div class="form-group">
                            <label for="language">Language</label>
                            <input type="text" id="language" name="language" value="<?php echo htmlspecialchars($event['language'] ?? ''); ?>" placeholder="e.g., English, Hindi">
                        </div>
                        
                        <div class="form-group">
                            <label for="genre">Genre</label>
                            <input type="text" id="genre" name="genre" value="<?php echo htmlspecialchars($event['genre'] ?? ''); ?>" placeholder="e.g., Rock, Pop, Classical">
                        </div>
                        
                        <div class="form-group">
                            <label for="refund_policy">Refund Policy</label>
                            <textarea id="refund_policy" name="refund_policy" rows="3" placeholder="Describe your refund policy..."><?php echo htmlspecialchars($event['refund_policy'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="age_restriction">Age Restriction</label>
                            <input type="text" id="age_restriction" name="age_restriction" value="<?php echo htmlspecialchars($event['age_restriction'] ?? ''); ?>" placeholder="e.g., 18+, All Ages">
                        </div>
                        
                        <div class="form-group">
                            <label for="artists">Artists (Optional)</label>
                            <select id="artists" name="artists[]" multiple class="form-control" style="min-height: 120px;">
                                <option value="">Loading artists...</option>
                            </select>
                            <small class="form-text">Hold Ctrl/Cmd to select multiple artists</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="sponsors">Sponsors (Optional)</label>
                            <select id="sponsors" name="sponsors[]" multiple class="form-control" style="min-height: 120px;">
                                <option value="">Loading sponsors...</option>
                            </select>
                            <small class="form-text">Hold Ctrl/Cmd to select multiple sponsors</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="agendas">Agendas (Optional)</label>
                            <select id="agendas" name="agendas[]" multiple class="form-control" style="min-height: 120px;">
                                <option value="">Loading agendas...</option>
                            </select>
                            <small class="form-text">Hold Ctrl/Cmd to select multiple agenda items</small>
                        </div>
                        
                        <div style="display: flex; gap: 1rem;">
                            <button type="submit" class="btn-sm btn-primary">Update Event</button>
                            <a href="events.php" class="btn-sm btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        const BASE_URL = '<?php echo $baseUrl; ?>';
        const EVENT_ID = <?php echo $event['id']; ?>;
    </script>
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    // Load artists and current event artists
    async function loadArtists() {
        try {
            // Load all available artists
            const artistsResponse = await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?status=all');
            const artistsData = await artistsResponse.json();
            
            // Load current event artists
            const eventArtistsResponse = await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?event_id=' + EVENT_ID);
            const eventArtistsData = await eventArtistsResponse.json();
            
            const select = document.getElementById('artists');
            const selectedIds = eventArtistsData.success && eventArtistsData.data 
                ? eventArtistsData.data.map(a => a.id.toString())
                : [];
            
            if (artistsData.success && artistsData.data) {
                select.innerHTML = '<option value=\"\">Select artists (optional)</option>';
                artistsData.data.forEach(artist => {
                    const option = document.createElement('option');
                    option.value = artist.id;
                    option.textContent = artist.name;
                    if (selectedIds.includes(artist.id.toString())) {
                        option.selected = true;
                    }
                    select.appendChild(option);
                });
            } else {
                select.innerHTML = '<option value=\"\">No artists available</option>';
            }
        } catch (error) {
            console.error('Error loading artists:', error);
            document.getElementById('artists').innerHTML = '<option value=\"\">Error loading artists</option>';
        }
    }
    
    // Load sponsors and current event sponsors
    async function loadSponsors() {
        try {
            // Load all available sponsors
            const sponsorsResponse = await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?status=all');
            const sponsorsData = await sponsorsResponse.json();
            
            // Load current event sponsors
            const eventSponsorsResponse = await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?event_id=' + EVENT_ID);
            const eventSponsorsData = await eventSponsorsResponse.json();
            
            const select = document.getElementById('sponsors');
            const selectedIds = eventSponsorsData.success && eventSponsorsData.data 
                ? eventSponsorsData.data.map(s => s.id.toString())
                : [];
            
            if (sponsorsData.success && sponsorsData.data) {
                select.innerHTML = '<option value=\"\">Select sponsors (optional)</option>';
                sponsorsData.data.forEach(sponsor => {
                    const option = document.createElement('option');
                    option.value = sponsor.id;
                    option.textContent = sponsor.name;
                    if (selectedIds.includes(sponsor.id.toString())) {
                        option.selected = true;
                    }
                    select.appendChild(option);
                });
            } else {
                select.innerHTML = '<option value=\"\">No sponsors available</option>';
            }
        } catch (error) {
            console.error('Error loading sponsors:', error);
            document.getElementById('sponsors').innerHTML = '<option value=\"\">Error loading sponsors</option>';
        }
    }
    
    // Load agendas and current event agendas
    async function loadAgendas() {
        try {
            // Load all available agendas
            const agendasResponse = await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?status=all');
            const agendasData = await agendasResponse.json();
            
            // Load current event agendas
            const eventAgendasResponse = await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?event_id=' + EVENT_ID);
            const eventAgendasData = await eventAgendasResponse.json();
            
            const select = document.getElementById('agendas');
            const selectedIds = eventAgendasData.success && eventAgendasData.data 
                ? eventAgendasData.data.map(a => a.id.toString())
                : [];
            
            if (agendasData.success && agendasData.data) {
                select.innerHTML = '<option value=\"\">Select agendas (optional)</option>';
                agendasData.data.forEach(agenda => {
                    const option = document.createElement('option');
                    option.value = agenda.id;
                    option.textContent = agenda.title;
                    if (selectedIds.includes(agenda.id.toString())) {
                        option.selected = true;
                    }
                    select.appendChild(option);
                });
            } else {
                select.innerHTML = '<option value=\"\">No agendas available</option>';
            }
        } catch (error) {
            console.error('Error loading agendas:', error);
            document.getElementById('agendas').innerHTML = '<option value=\"\">Error loading agendas</option>';
        }
    }
    
    // Load artists, sponsors & agendas on page load
    loadArtists();
    loadSponsors();
    loadAgendas();
    
    document.getElementById('editEventForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = {};
        formData.forEach((value, key) => {
            if (key === 'artists[]') {
                if (!data.artists) data.artists = [];
                data.artists.push(value);
            } else if (key === 'sponsors[]') {
                if (!data.sponsors) data.sponsors = [];
                data.sponsors.push(value);
            } else if (key === 'agendas[]') {
                if (!data.agendas) data.agendas = [];
                data.agendas.push(value);
            } else {
                data[key] = value;
            }
        });
        
        // Remove artists/sponsors/agendas from event data (will be handled separately)
        const artistIds = data.artists || [];
        const sponsorIds = data.sponsors || [];
        const agendaIds = data.agendas || [];
        delete data.artists;
        delete data.sponsors;
        delete data.agendas;
        
        try {
            // Update event
            const response = await fetch(this.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'Failed to update event');
            }
            
            // Get current event artists
            const currentResponse = await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?event_id=' + EVENT_ID);
            const currentData = await currentResponse.json();
            const currentArtistIds = currentData.success && currentData.data 
                ? currentData.data.map(a => a.id.toString())
                : [];
            
            // Remove artists not in new selection
            for (const currentId of currentArtistIds) {
                if (!artistIds.includes(currentId)) {
                    await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?action=unlink', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        credentials: 'same-origin',
                        body: JSON.stringify({
                            event_id: EVENT_ID,
                            artist_id: parseInt(currentId)
                        })
                    });
                }
            }
            
            // Add new artists
            for (const newId of artistIds) {
                if (!currentArtistIds.includes(newId)) {
                    await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?action=link', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        credentials: 'same-origin',
                        body: JSON.stringify({
                            event_id: EVENT_ID,
                            artist_ids: [parseInt(newId)]
                        })
                    });
                }
            }
            
            // Get current event sponsors
            const currentSponsorsResponse = await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?event_id=' + EVENT_ID);
            const currentSponsorsData = await currentSponsorsResponse.json();
            const currentSponsorIds = currentSponsorsData.success && currentSponsorsData.data 
                ? currentSponsorsData.data.map(s => s.id.toString())
                : [];
            
            // Remove sponsors not in new selection
            for (const currentId of currentSponsorIds) {
                if (!sponsorIds.includes(currentId)) {
                    await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?action=unlink', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        credentials: 'same-origin',
                        body: JSON.stringify({
                            event_id: EVENT_ID,
                            sponsor_id: parseInt(currentId)
                        })
                    });
                }
            }
            
            // Add new sponsors
            for (const newId of sponsorIds) {
                if (!currentSponsorIds.includes(newId)) {
                    await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?action=link', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        credentials: 'same-origin',
                        body: JSON.stringify({
                            event_id: EVENT_ID,
                            sponsor_ids: [parseInt(newId)]
                        })
                    });
                }
            }
            
            // Get current event agendas
            const currentAgendasResponse = await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?event_id=' + EVENT_ID);
            const currentAgendasData = await currentAgendasResponse.json();
            const currentAgendaIds = currentAgendasData.success && currentAgendasData.data 
                ? currentAgendasData.data.map(a => a.id.toString())
                : [];
            
            // Remove agendas not in new selection
            for (const currentId of currentAgendaIds) {
                if (!agendaIds.includes(currentId)) {
                    await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?action=unlink', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        credentials: 'same-origin',
                        body: JSON.stringify({
                            event_id: EVENT_ID,
                            agenda_id: parseInt(currentId)
                        })
                    });
                }
            }
            
            // Add new agendas
            for (const newId of agendaIds) {
                if (!currentAgendaIds.includes(newId)) {
                    await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?action=link', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        credentials: 'same-origin',
                        body: JSON.stringify({
                            event_id: EVENT_ID,
                            agenda_ids: [parseInt(newId)]
                        })
                    });
                }
            }
            
            alert('Event updated successfully');
            window.location.href = BASE_URL + '/organizer/pages/events.php';
            
        } catch (error) {
            console.error('Error:', error);
            alert('Error: ' + error.message);
        }
    });
    </script>
</body>
</html>

