<?php
/**
 * Organizer - Edit Artist
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to edit artists
if (!hasPermission('organizer.artists.edit')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to edit artists.'));
    exit;
}

$artistId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$artistId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/artists.php?error=' . urlencode('Artist ID is required.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Verify ownership and fetch artist
try {
    $check = $pdo->query("SHOW COLUMNS FROM artists LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

$sql = "SELECT * FROM artists WHERE id = :id AND organizer_id = :organizer_id";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
$artist = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$artist) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/artists.php?error=' . urlencode('Artist not found or you do not have permission to edit this artist.'));
    exit;
}

// Decode social links
$socialLinks = [];
if (!empty($artist['social_links'])) {
    $socialLinks = json_decode($artist['social_links'], true) ?: [];
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Artist | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'artists';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Edit Artist</h1>
                
                <div class="section">
                    <form id="editArtistForm" method="POST" action="<?php echo $baseUrl; ?>/api/organizer/artistsapi.php?action=update&id=<?php echo $artist['id']; ?>">
                        <div class="form-group">
                            <label for="name">Artist Name *</label>
                            <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($artist['name']); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="bio">Bio / Description</label>
                            <textarea id="bio" name="bio" rows="6" placeholder="Tell us about this artist..."><?php echo htmlspecialchars($artist['bio'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="profile_image">Profile Image URL</label>
                            <input type="url" id="profile_image" name="profile_image" value="<?php echo htmlspecialchars($artist['profile_image'] ?? ''); ?>" placeholder="https://example.com/image.jpg">
                            <small class="form-text">Enter a URL to the artist's profile image</small>
                        </div>
                        
                        <div class="form-group">
                            <label>Social Links (Optional)</label>
                            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-top: 0.5rem;">
                                <div>
                                    <label for="social_website" style="font-size: 0.875rem; color: #6b7280;">Website URL</label>
                                    <input type="url" id="social_website" name="social_links[website]" value="<?php echo htmlspecialchars($socialLinks['website'] ?? ''); ?>" placeholder="https://example.com">
                                </div>
                                <div>
                                    <label for="social_facebook" style="font-size: 0.875rem; color: #6b7280;">Facebook URL</label>
                                    <input type="url" id="social_facebook" name="social_links[facebook]" value="<?php echo htmlspecialchars($socialLinks['facebook'] ?? ''); ?>" placeholder="https://facebook.com/...">
                                </div>
                                <div>
                                    <label for="social_twitter" style="font-size: 0.875rem; color: #6b7280;">Twitter/X URL</label>
                                    <input type="url" id="social_twitter" name="social_links[twitter]" value="<?php echo htmlspecialchars($socialLinks['twitter'] ?? ''); ?>" placeholder="https://twitter.com/...">
                                </div>
                                <div>
                                    <label for="social_instagram" style="font-size: 0.875rem; color: #6b7280;">Instagram URL</label>
                                    <input type="url" id="social_instagram" name="social_links[instagram]" value="<?php echo htmlspecialchars($socialLinks['instagram'] ?? ''); ?>" placeholder="https://instagram.com/...">
                                </div>
                                <div>
                                    <label for="social_youtube" style="font-size: 0.875rem; color: #6b7280;">YouTube URL</label>
                                    <input type="url" id="social_youtube" name="social_links[youtube]" value="<?php echo htmlspecialchars($socialLinks['youtube'] ?? ''); ?>" placeholder="https://youtube.com/...">
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="status">Status</label>
                            <select id="status" name="status">
                                <option value="unpublished" <?php echo ($artist['status'] === 'unpublished') ? 'selected' : ''; ?>>Unpublished</option>
                                <option value="published" <?php echo ($artist['status'] === 'published') ? 'selected' : ''; ?>>Published</option>
                            </select>
                            <small class="form-text">Unpublished artists are not visible to the public</small>
                        </div>
                        
                        <div style="display: flex; gap: 1rem;">
                            <button type="submit" class="btn-sm btn-primary">Update Artist</button>
                            <a href="artists.php" class="btn-sm btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        const BASE_URL = '<?php echo $baseUrl; ?>';
    </script>
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    document.getElementById('editArtistForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = {};
        
        // Collect form data
        for (let [key, value] of formData.entries()) {
            if (key.startsWith('social_links[')) {
                const socialKey = key.match(/\[(.*?)\]/)[1];
                if (!data.social_links) data.social_links = {};
                if (value.trim()) {
                    data.social_links[socialKey] = value.trim();
                }
            } else {
                if (value.trim()) {
                    data[key] = value.trim();
                }
            }
        }
        
        try {
            const response = await fetch(this.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'Failed to update artist');
            }
            
            alert('Artist updated successfully');
            window.location.href = BASE_URL + '/organizer/pages/artists.php';
            
        } catch (error) {
            console.error('Error updating artist:', error);
            alert('Error: ' + error.message);
        }
    });
    </script>
</body>
</html>
