<?php
/**
 * Organizer - Edit Agenda
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to edit agendas
if (!hasPermission('organizer.agendas.edit')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to edit agendas.'));
    exit;
}

$agendaId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$agendaId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/agendas.php?error=' . urlencode('Agenda ID is required.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility
try {
    $check = $pdo->query("SHOW COLUMNS FROM agendas LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Verify ownership and fetch agenda
$sql = "SELECT * FROM agendas WHERE id = :id AND organizer_id = :organizer_id";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
$agenda = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$agenda) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/agendas.php?error=' . urlencode('Agenda not found or you do not have permission to edit this agenda.'));
    exit;
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Agenda | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'agendas';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Edit Agenda</h1>
                
                <div class="section">
                    <form id="editAgendaForm" method="POST" action="<?php echo $baseUrl; ?>/api/organizer/agendasapi.php?action=update&id=<?php echo $agenda['id']; ?>">
                        <div class="form-group">
                            <label for="title">Title *</label>
                            <input type="text" id="title" name="title" value="<?php echo htmlspecialchars($agenda['title']); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea id="description" name="description" rows="4" placeholder="Describe this agenda item..."><?php echo htmlspecialchars($agenda['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="start_time">Start Time</label>
                                <input type="datetime-local" id="start_time" name="start_time" value="<?php echo $agenda['start_time'] ? date('Y-m-d\TH:i', strtotime($agenda['start_time'])) : ''; ?>">
                            </div>
                            <div class="form-group">
                                <label for="end_time">End Time</label>
                                <input type="datetime-local" id="end_time" name="end_time" value="<?php echo $agenda['end_time'] ? date('Y-m-d\TH:i', strtotime($agenda['end_time'])) : ''; ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="status">Status</label>
                            <select id="status" name="status">
                                <option value="unpublished" <?php echo ($agenda['status'] === 'unpublished') ? 'selected' : ''; ?>>Unpublished</option>
                                <option value="published" <?php echo ($agenda['status'] === 'published') ? 'selected' : ''; ?>>Published</option>
                            </select>
                            <small class="form-text">Unpublished agendas are not visible publicly</small>
                        </div>
                        
                        <div style="display: flex; gap: 1rem;">
                            <button type="submit" class="btn-sm btn-primary">Update Agenda</button>
                            <a href="agendas.php" class="btn-sm btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        const BASE_URL = '<?php echo $baseUrl; ?>';
    </script>
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    document.getElementById('editAgendaForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = {};
        
        for (let [key, value] of formData.entries()) {
            if (value.trim()) {
                data[key] = value.trim();
            }
        }
        
        try {
            const response = await fetch(this.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'Failed to update agenda');
            }
            
            alert('Agenda updated successfully');
            window.location.href = BASE_URL + '/organizer/pages/agendas.php';
            
        } catch (error) {
            console.error('Error updating agenda:', error);
            alert('Error: ' + error.message);
        }
    });
    </script>
</body>
</html>


