<?php
/**
 * Organizer - Create Event
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to create events
if (!hasPermission('organizer.events.create')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to create events. Please contact your administrator.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Get categories
$stmt = $pdo->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY name ASC");
$categories = $stmt->fetchAll();

// Get venues
$stmt = $pdo->query("SELECT * FROM venues WHERE is_active = 1 ORDER BY venue_name ASC");
$venues = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Event | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'create-event';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Create New Event</h1>
                
                <div class="section">
                    <form id="createEventForm" method="POST" action="<?php echo $baseUrl; ?>/api/organizer/createeventapi.php">
                        <div class="form-group">
                            <label for="name">Event Name *</label>
                            <input type="text" id="name" name="name" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="short_description">Short Description</label>
                            <input type="text" id="short_description" name="short_description" maxlength="500">
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Full Description *</label>
                            <textarea id="description" name="description" rows="6" required></textarea>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="event_type">Event Type</label>
                                <select id="event_type" name="event_type">
                                    <option value="">Select Type</option>
                                    <option value="conference">Conference</option>
                                    <option value="exhibition">Exhibition</option>
                                    <option value="festival">Festival</option>
                                    <option value="concert">Concert</option>
                                    <option value="workshop">Workshop</option>
                                    <option value="summit">Summit</option>
                                    <option value="trade-show">Trade Show</option>
                                    <option value="sports">Sports</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="category">Category</label>
                                <select id="category" name="category">
                                    <option value="">Select Category</option>
                                    <?php foreach ($categories as $cat): ?>
                                        <option value="<?php echo htmlspecialchars($cat['slug']); ?>"><?php echo htmlspecialchars($cat['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="start_date">Start Date & Time *</label>
                                <input type="datetime-local" id="start_date" name="start_date" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="end_date">End Date & Time *</label>
                                <input type="datetime-local" id="end_date" name="end_date" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="venue_id">Venue</label>
                            <select id="venue_id" name="venue_id">
                                <option value="">Select Venue (Optional)</option>
                                <?php foreach ($venues as $venue): ?>
                                    <option value="<?php echo $venue['id']; ?>"><?php echo htmlspecialchars($venue['venue_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 1rem;">
                            <div class="form-group">
                                <label for="city">City</label>
                                <input type="text" id="city" name="city">
                            </div>
                            
                            <div class="form-group">
                                <label for="state">State</label>
                                <input type="text" id="state" name="state">
                            </div>
                            
                            <div class="form-group">
                                <label for="country">Country *</label>
                                <input type="text" id="country" name="country" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="address">Address</label>
                            <textarea id="address" name="address" rows="2"></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="banner_image">Banner Image URL</label>
                            <input type="url" id="banner_image" name="banner_image" placeholder="https://example.com/image.jpg">
                        </div>
                        
                        <div class="form-group">
                            <label for="website_url">Event Website URL</label>
                            <input type="url" id="website_url" name="website_url" placeholder="https://example.com">
                        </div>
                        
                        <div class="form-group">
                            <label for="attendees_range">Expected Attendees</label>
                            <input type="text" id="attendees_range" name="attendees_range" placeholder="e.g., 1000+, 500-1000">
                        </div>
                        
                        <div class="form-group">
                            <label for="refund_policy">Refund Policy</label>
                            <textarea id="refund_policy" name="refund_policy" rows="3" placeholder="Describe your refund policy..."></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="age_restriction">Age Restriction</label>
                            <input type="text" id="age_restriction" name="age_restriction" placeholder="e.g., 18+, All Ages">
                        </div>
                        
                        <div class="form-group">
                            <label for="artists">Artists (Optional)</label>
                            <select id="artists" name="artists[]" multiple class="form-control" style="min-height: 120px;">
                                <option value="">Loading artists...</option>
                            </select>
                            <small class="form-text">Hold Ctrl/Cmd to select multiple artists</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="sponsors">Sponsors (Optional)</label>
                            <select id="sponsors" name="sponsors[]" multiple class="form-control" style="min-height: 120px;">
                                <option value="">Loading sponsors...</option>
                            </select>
                            <small class="form-text">Hold Ctrl/Cmd to select multiple sponsors</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="agendas">Agendas (Optional)</label>
                            <select id="agendas" name="agendas[]" multiple class="form-control" style="min-height: 120px;">
                                <option value="">Loading agendas...</option>
                            </select>
                            <small class="form-text">Hold Ctrl/Cmd to select multiple agenda items</small>
                        </div>
                        
                        <button type="submit" class="btn-sm btn-primary">Create Event</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        const BASE_URL = '<?php echo $baseUrl; ?>';
    </script>
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    // Load artists for dropdown
    async function loadArtists() {
        try {
            const response = await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?status=all');
            const data = await response.json();
            
            const select = document.getElementById('artists');
            if (data.success && data.data) {
                select.innerHTML = '<option value=\"\">Select artists (optional)</option>';
                data.data.forEach(artist => {
                    const option = document.createElement('option');
                    option.value = artist.id;
                    option.textContent = artist.name;
                    select.appendChild(option);
                });
            } else {
                select.innerHTML = '<option value=\"\">No artists available</option>';
            }
        } catch (error) {
            console.error('Error loading artists:', error);
            document.getElementById('artists').innerHTML = '<option value=\"\">Error loading artists</option>';
        }
    }
    
    // Load sponsors for dropdown
    async function loadSponsors() {
        try {
            const response = await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?status=all');
            const data = await response.json();
            
            const select = document.getElementById('sponsors');
            if (data.success && data.data) {
                select.innerHTML = '<option value=\"\">Select sponsors (optional)</option>';
                data.data.forEach(sponsor => {
                    const option = document.createElement('option');
                    option.value = sponsor.id;
                    option.textContent = sponsor.name;
                    select.appendChild(option);
                });
            } else {
                select.innerHTML = '<option value=\"\">No sponsors available</option>';
            }
        } catch (error) {
            console.error('Error loading sponsors:', error);
            document.getElementById('sponsors').innerHTML = '<option value=\"\">Error loading sponsors</option>';
        }
    }
    
    // Load agendas for dropdown
    async function loadAgendas() {
        try {
            const response = await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?status=all');
            const data = await response.json();
            
            const select = document.getElementById('agendas');
            if (data.success && data.data) {
                select.innerHTML = '<option value=\"\">Select agendas (optional)</option>';
                data.data.forEach(agenda => {
                    const option = document.createElement('option');
                    option.value = agenda.id;
                    option.textContent = agenda.title;
                    select.appendChild(option);
                });
            } else {
                select.innerHTML = '<option value=\"\">No agendas available</option>';
            }
        } catch (error) {
            console.error('Error loading agendas:', error);
            document.getElementById('agendas').innerHTML = '<option value=\"\">Error loading agendas</option>';
        }
    }
    
    // Load artists, sponsors & agendas on page load
    loadArtists();
    loadSponsors();
    loadAgendas();
    
    document.getElementById('createEventForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = {};
        formData.forEach((value, key) => {
            if (key === 'artists[]') {
                if (!data.artists) data.artists = [];
                data.artists.push(value);
            } else if (key === 'sponsors[]') {
                if (!data.sponsors) data.sponsors = [];
                data.sponsors.push(value);
            } else if (key === 'agendas[]') {
                if (!data.agendas) data.agendas = [];
                data.agendas.push(value);
            } else {
                data[key] = value;
            }
        });
        
        // Remove artists/sponsors/agendas from event data (will be handled separately)
        const artistIds = data.artists || [];
        const sponsorIds = data.sponsors || [];
        const agendaIds = data.agendas || [];
        delete data.artists;
        delete data.sponsors;
        delete data.agendas;
        
        try {
            // Create event
            const response = await fetch(this.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'Failed to create event');
            }
            
            // Link artists if any selected
            if (artistIds.length > 0 && result.event_id) {
                await fetch(BASE_URL + '/api/organizer/eventartistsapi.php?action=link', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    credentials: 'same-origin',
                    body: JSON.stringify({
                        event_id: result.event_id,
                        artist_ids: artistIds.map(id => parseInt(id))
                    })
                });
            }
            
            // Link sponsors if any selected
            if (sponsorIds.length > 0 && result.event_id) {
                await fetch(BASE_URL + '/api/organizer/eventsponsorsapi.php?action=link', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    credentials: 'same-origin',
                    body: JSON.stringify({
                        event_id: result.event_id,
                        sponsor_ids: sponsorIds.map(id => parseInt(id))
                    })
                });
            }
            
            // Link agendas if any selected
            if (agendaIds.length > 0 && result.event_id) {
                await fetch(BASE_URL + '/api/organizer/eventagendasapi.php?action=link', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    credentials: 'same-origin',
                    body: JSON.stringify({
                        event_id: result.event_id,
                        agenda_ids: agendaIds.map(id => parseInt(id))
                    })
                });
            }
            
            alert('Event created successfully');
            window.location.href = BASE_URL + '/organizer/pages/events.php';
            
        } catch (error) {
            console.error('Error:', error);
            alert('Error: ' + error.message);
        }
    });
    </script>
</body>
</html>

