<?php
/**
 * Organizer - Attendees Management
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view attendees
if (!hasPermission('organizer.attendees.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view attendees.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Get filter parameters
$event_id = $_GET['event_id'] ?? null;
$search = $_GET['search'] ?? '';

// Get organizer's events
$stmt = $pdo->prepare("SELECT id, name, start_date FROM events WHERE organizer_id = :organizer_id ORDER BY start_date DESC");
$stmt->execute([':organizer_id' => $organizer['id']]);
$events = $stmt->fetchAll();

// Build query for attendees
$where = ["e.organizer_id = :organizer_id"];
$params = [':organizer_id' => $organizer['id']];

if ($event_id) {
    $where[] = "e.id = :event_id";
    $params[':event_id'] = $event_id;
}

if ($search) {
    $where[] = "(c.full_name LIKE :search OR c.email LIKE :search OR t.ticket_number LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

$whereClause = 'WHERE ' . implode(' AND ', $where);

// Get attendees
$stmt = $pdo->prepare("
    SELECT 
        t.*,
        b.booking_reference,
        e.name as event_name,
        e.start_date as event_date,
        c.full_name as customer_name,
        c.email as customer_email,
        c.phone as customer_phone,
        tt.name as ticket_type_name
    FROM tickets t
    INNER JOIN bookings b ON t.booking_id = b.id
    INNER JOIN events e ON b.event_id = e.id
    INNER JOIN customers c ON b.customer_id = c.id
    INNER JOIN ticket_types tt ON t.ticket_type_id = tt.id
    {$whereClause}
    ORDER BY t.created_at DESC
    LIMIT 100
");
$stmt->execute($params);
$attendees = $stmt->fetchAll();

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Attendees | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'attendees';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <h1>Attendees Management</h1>
                
                <!-- Filters -->
                <div class="section">
                    <form method="GET" style="display: flex; gap: 1rem; align-items: end;">
                        <div class="form-group" style="flex: 1;">
                            <label>Search</label>
                            <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search by name, email, or ticket number...">
                        </div>
                        <div class="form-group" style="width: 250px;">
                            <label>Event</label>
                            <select name="event_id">
                                <option value="">All Events</option>
                                <?php foreach ($events as $event): ?>
                                    <option value="<?php echo $event['id']; ?>" <?php echo $event_id == $event['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($event['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group" style="width: auto;">
                            <button type="submit" class="btn-sm btn-primary">Filter</button>
                        </div>
                    </form>
                </div>
                
                <!-- Attendees Table -->
                <div class="section">
                    <h2>Attendees List</h2>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Ticket Number</th>
                                    <th>Event</th>
                                    <th>Customer</th>
                                    <th>Ticket Type</th>
                                    <th>Check-in Status</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($attendees)): ?>
                                <tr>
                                    <td colspan="7" style="text-align: center; padding: 2rem;">
                                        No attendees found.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($attendees as $attendee): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($attendee['ticket_number']); ?></td>
                                        <td><?php echo htmlspecialchars($attendee['event_name']); ?></td>
                                        <td>
                                            <div><?php echo htmlspecialchars($attendee['customer_name'] ?? 'N/A'); ?></div>
                                            <div style="font-size: 0.875rem; color: #6b7280;"><?php echo htmlspecialchars($attendee['customer_email']); ?></div>
                                        </td>
                                        <td><?php echo htmlspecialchars($attendee['ticket_type_name']); ?></td>
                                        <td>
                                            <?php if ($attendee['checked_in']): ?>
                                                <span class="status-badge status-approved">Checked In</span>
                                                <div style="font-size: 0.75rem; color: #6b7280;">
                                                    <?php echo date('M d, Y H:i', strtotime($attendee['checked_in_at'])); ?>
                                                </div>
                                            <?php else: ?>
                                                <span class="status-badge status-pending">Not Checked In</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><span class="status-badge status-<?php echo $attendee['status']; ?>"><?php echo ucfirst($attendee['status']); ?></span></td>
                                        <td>
                                            <a href="attendee-detail.php?id=<?php echo $attendee['id']; ?>" class="btn-sm btn-primary">View</a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
</body>
</html>

