<?php
/**
 * Organizer - My Artists
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view artists
if (!hasPermission('organizer.artists.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view artists.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility
try {
    $check = $pdo->query("SHOW COLUMNS FROM artists LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Get status filter from URL
$statusFilter = $_GET['status'] ?? 'all';
$searchQuery = $_GET['search'] ?? '';

// Get artists (exclude soft-deleted if supported)
$sql = "
    SELECT * FROM artists
    WHERE organizer_id = :organizer_id
";
$params = [':organizer_id' => $organizer['id']];

if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

// Apply status filter
if ($statusFilter !== 'all') {
    $sql .= " AND status = :status";
    $params[':status'] = $statusFilter;
}

// Apply search filter
if (!empty($searchQuery)) {
    $sql .= " AND name LIKE :search";
    $params[':search'] = '%' . $searchQuery . '%';
}

$sql .= " ORDER BY created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$artists = $stmt->fetchAll(PDO::FETCH_ASSOC);

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Artists | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'artists';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                    <h1>My Artists</h1>
                    <?php if (hasPermission('organizer.artists.create')): ?>
                    <a href="create-artist.php" class="btn-sm btn-primary">+ Create Artist</a>
                    <?php endif; ?>
                </div>
                
                <!-- Filters -->
                <div class="section" style="margin-bottom: 1.5rem;">
                    <form method="GET" action="" style="display: flex; gap: 1rem; align-items: flex-end; flex-wrap: wrap;">
                        <div class="form-group" style="margin-bottom: 0;">
                            <label for="status">Status:</label>
                            <select id="status" name="status" style="min-width: 150px;">
                                <option value="all" <?php echo $statusFilter === 'all' ? 'selected' : ''; ?>>All</option>
                                <option value="published" <?php echo $statusFilter === 'published' ? 'selected' : ''; ?>>Published</option>
                                <option value="unpublished" <?php echo $statusFilter === 'unpublished' ? 'selected' : ''; ?>>Unpublished</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 0;">
                            <label for="search">Search:</label>
                            <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($searchQuery); ?>" placeholder="Search by name...">
                        </div>
                        <button type="submit" class="btn-sm" style="background: #374151; color: white; border: none;">Apply Filters</button>
                        <?php if ($statusFilter !== 'all' || !empty($searchQuery)): ?>
                        <a href="artists.php" class="btn-sm btn-secondary">Clear</a>
                        <?php endif; ?>
                    </form>
                </div>
                
                <div class="section">
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Image</th>
                                    <th>Name</th>
                                    <th>Bio</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($artists)): ?>
                                <tr>
                                    <td colspan="6" style="text-align: center; padding: 2rem; color: #6b7280;">
                                        No artists found. <?php if (hasPermission('organizer.artists.create')): ?><a href="create-artist.php">Create your first artist</a><?php endif; ?>
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($artists as $artist): ?>
                                    <tr>
                                        <td>
                                            <?php if ($artist['profile_image']): ?>
                                                <img src="<?php echo htmlspecialchars($artist['profile_image']); ?>" alt="<?php echo htmlspecialchars($artist['name']); ?>" style="width: 50px; height: 50px; object-fit: cover; border-radius: 8px;">
                                            <?php else: ?>
                                                <div style="width: 50px; height: 50px; background: #e5e7eb; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 20px;">🎤</div>
                                            <?php endif; ?>
                                        </td>
                                        <td><strong><?php echo htmlspecialchars($artist['name']); ?></strong></td>
                                        <td>
                                            <?php if ($artist['bio']): ?>
                                                <?php echo htmlspecialchars(mb_substr($artist['bio'], 0, 100)) . (mb_strlen($artist['bio']) > 100 ? '...' : ''); ?>
                                            <?php else: ?>
                                                <em style="color: #9ca3af;">No bio</em>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($artist['status'] === 'published'): ?>
                                                <span class="status-badge status-published">Published</span>
                                            <?php else: ?>
                                                <span class="status-badge status-unpublished">Unpublished</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('M d, Y', strtotime($artist['created_at'])); ?></td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem;">
                                                <a href="artist-detail.php?id=<?php echo $artist['id']; ?>" class="btn-sm btn-primary">View</a>
                                                <?php if (hasPermission('organizer.artists.edit')): ?>
                                                <a href="edit-artist.php?id=<?php echo $artist['id']; ?>" class="btn-sm btn-secondary">Edit</a>
                                                <?php endif; ?>
                                                <?php if (hasPermission('organizer.artists.delete')): ?>
                                                <button onclick="deleteArtist(<?php echo $artist['id']; ?>, '<?php echo htmlspecialchars(addslashes($artist['name'])); ?>')" class="btn-sm" style="background: #dc2626; color: white; border: none; cursor: pointer;">Delete</button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    function deleteArtist(artistId, artistName) {
        if (!confirm('Are you sure you want to delete "' + artistName + '"? This action cannot be undone.')) {
            return;
        }
        
        fetch('<?php echo $baseUrl; ?>/api/organizer/artistsapi.php?action=delete&id=' + artistId, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Artist deleted successfully');
                location.reload();
            } else {
                alert('Error: ' + (data.message || 'Failed to delete artist'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while deleting the artist');
        });
    }
    </script>
</body>
</html>
