<?php
/**
 * Organizer - Artist Detail
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view artists
if (!hasPermission('organizer.artists.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view artists.'));
    exit;
}

$artistId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$artistId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/artists.php?error=' . urlencode('Artist ID is required.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Verify ownership and fetch artist
try {
    $check = $pdo->query("SHOW COLUMNS FROM artists LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

$sql = "SELECT * FROM artists WHERE id = :id AND organizer_id = :organizer_id";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
$artist = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$artist) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/artists.php?error=' . urlencode('Artist not found or you do not have permission to view this artist.'));
    exit;
}

// Decode social links
$socialLinks = [];
if (!empty($artist['social_links'])) {
    $socialLinks = json_decode($artist['social_links'], true) ?: [];
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($artist['name']); ?> | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'artists';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="page-header">
                <div>
                    <h1><?php echo htmlspecialchars($artist['name']); ?></h1>
                    <p>Artist Details</p>
                </div>
                <div>
                    <?php if (hasPermission('organizer.artists.edit')): ?>
                    <a href="<?php echo $baseUrl; ?>/organizer/pages/edit-artist.php?id=<?php echo $artist['id']; ?>" class="btn btn-primary">Edit Artist</a>
                    <?php endif; ?>
                    <a href="<?php echo $baseUrl; ?>/organizer/pages/artists.php" class="btn btn-secondary">Back to Artists</a>
                </div>
            </div>
            
            <div class="detail-container">
                <div class="detail-section">
                    <div class="detail-row">
                        <div class="detail-label">Profile Image:</div>
                        <div class="detail-value">
                            <?php if ($artist['profile_image']): ?>
                                <img src="<?php echo htmlspecialchars($artist['profile_image']); ?>" alt="<?php echo htmlspecialchars($artist['name']); ?>" style="max-width: 200px; border-radius: 8px;">
                            <?php else: ?>
                                <div style="width: 200px; height: 200px; background: #e5e7eb; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 60px;">🎤</div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="detail-row">
                        <div class="detail-label">Name:</div>
                        <div class="detail-value"><strong><?php echo htmlspecialchars($artist['name']); ?></strong></div>
                    </div>
                    
                    <div class="detail-row">
                        <div class="detail-label">Status:</div>
                        <div class="detail-value">
                            <?php if ($artist['status'] === 'published'): ?>
                                <span class="badge badge-success">Published</span>
                            <?php else: ?>
                                <span class="badge badge-warning">Unpublished</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if ($artist['bio']): ?>
                    <div class="detail-row">
                        <div class="detail-label">Bio:</div>
                        <div class="detail-value"><?php echo nl2br(htmlspecialchars($artist['bio'])); ?></div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($socialLinks)): ?>
                    <div class="detail-row">
                        <div class="detail-label">Social Links:</div>
                        <div class="detail-value">
                            <div class="social-links">
                                <?php if (!empty($socialLinks['website'])): ?>
                                    <a href="<?php echo htmlspecialchars($socialLinks['website']); ?>" target="_blank" class="social-link">🌐 Website</a>
                                <?php endif; ?>
                                <?php if (!empty($socialLinks['facebook'])): ?>
                                    <a href="<?php echo htmlspecialchars($socialLinks['facebook']); ?>" target="_blank" class="social-link">📘 Facebook</a>
                                <?php endif; ?>
                                <?php if (!empty($socialLinks['twitter'])): ?>
                                    <a href="<?php echo htmlspecialchars($socialLinks['twitter']); ?>" target="_blank" class="social-link">🐦 Twitter/X</a>
                                <?php endif; ?>
                                <?php if (!empty($socialLinks['instagram'])): ?>
                                    <a href="<?php echo htmlspecialchars($socialLinks['instagram']); ?>" target="_blank" class="social-link">📷 Instagram</a>
                                <?php endif; ?>
                                <?php if (!empty($socialLinks['youtube'])): ?>
                                    <a href="<?php echo htmlspecialchars($socialLinks['youtube']); ?>" target="_blank" class="social-link">📺 YouTube</a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <div class="detail-row">
                        <div class="detail-label">Created:</div>
                        <div class="detail-value"><?php echo date('F j, Y g:i A', strtotime($artist['created_at'])); ?></div>
                    </div>
                    
                    <?php if ($artist['updated_at'] !== $artist['created_at']): ?>
                    <div class="detail-row">
                        <div class="detail-label">Last Updated:</div>
                        <div class="detail-value"><?php echo date('F j, Y g:i A', strtotime($artist['updated_at'])); ?></div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
    
    <style>
        .detail-container {
            background: white;
            border-radius: 8px;
            padding: 2rem;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .detail-section {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
        }
        .detail-row {
            display: grid;
            grid-template-columns: 150px 1fr;
            gap: 1rem;
            align-items: start;
        }
        .detail-label {
            font-weight: 600;
            color: #374151;
        }
        .detail-value {
            color: #111827;
        }
        .social-links {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
        }
        .social-link {
            display: inline-block;
            padding: 0.5rem 1rem;
            background: #f3f4f6;
            border-radius: 6px;
            text-decoration: none;
            color: #111827;
            transition: background 0.2s;
        }
        .social-link:hover {
            background: #e5e7eb;
        }
        @media (max-width: 768px) {
            .detail-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</body>
</html>

