<?php
/**
 * Organizer - My Agendas
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view agendas
if (!hasPermission('organizer.agendas.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view agendas.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility
try {
    $check = $pdo->query("SHOW COLUMNS FROM agendas LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Get filters from URL
$statusFilter = $_GET['status'] ?? 'all';
$searchQuery = $_GET['search'] ?? '';

// Get agendas (exclude soft-deleted if supported)
$sql = "
    SELECT * FROM agendas
    WHERE organizer_id = :organizer_id
";
$params = [':organizer_id' => $organizer['id']];

if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

// Apply status filter
if ($statusFilter !== 'all') {
    $sql .= " AND status = :status";
    $params[':status'] = $statusFilter;
}

// Apply search filter
if (!empty($searchQuery)) {
    $sql .= " AND title LIKE :search";
    $params[':search'] = '%' . $searchQuery . '%';
}

$sql .= " ORDER BY COALESCE(start_time, created_at) ASC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$agendas = $stmt->fetchAll(PDO::FETCH_ASSOC);

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Agendas | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'agendas';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="admin-content">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                    <h1>My Agendas</h1>
                    <?php if (hasPermission('organizer.agendas.create')): ?>
                    <a href="create-agenda.php" class="btn-sm btn-primary">+ Create Agenda</a>
                    <?php endif; ?>
                </div>
                
                <!-- Filters -->
                <div class="section" style="margin-bottom: 1.5rem;">
                    <form method="GET" action="" style="display: flex; gap: 1rem; align-items: flex-end; flex-wrap: wrap;">
                        <div class="form-group" style="margin-bottom: 0;">
                            <label for="status">Status:</label>
                            <select id="status" name="status" style="min-width: 150px;">
                                <option value="all" <?php echo $statusFilter === 'all' ? 'selected' : ''; ?>>All</option>
                                <option value="published" <?php echo $statusFilter === 'published' ? 'selected' : ''; ?>>Published</option>
                                <option value="unpublished" <?php echo $statusFilter === 'unpublished' ? 'selected' : ''; ?>>Unpublished</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 0;">
                            <label for="search">Search:</label>
                            <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($searchQuery); ?>" placeholder="Search by title...">
                        </div>
                        <button type="submit" class="btn-sm" style="background: #374151; color: white; border: none;">Apply Filters</button>
                        <?php if ($statusFilter !== 'all' || !empty($searchQuery)): ?>
                        <a href="agendas.php" class="btn-sm btn-secondary">Clear</a>
                        <?php endif; ?>
                    </form>
                </div>
                
                <div class="section">
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Title</th>
                                    <th>Time</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($agendas)): ?>
                                <tr>
                                    <td colspan="5" style="text-align: center; padding: 2rem; color: #6b7280;">
                                        No agendas found. <?php if (hasPermission('organizer.agendas.create')): ?><a href="create-agenda.php">Create your first agenda</a><?php endif; ?>
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($agendas as $agenda): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($agenda['title']); ?></strong></td>
                                        <td>
                                            <?php if ($agenda['start_time'] || $agenda['end_time']): ?>
                                                <?php
                                                    $start = $agenda['start_time'] ? date('M d, Y H:i', strtotime($agenda['start_time'])) : '';
                                                    $end = $agenda['end_time'] ? date('M d, Y H:i', strtotime($agenda['end_time'])) : '';
                                                ?>
                                                <span><?php echo htmlspecialchars(trim($start . ($end ? ' - ' . $end : ''))); ?></span>
                                            <?php else: ?>
                                                <em style="color: #9ca3af;">No time set</em>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($agenda['status'] === 'published'): ?>
                                                <span class="status-badge status-published">Published</span>
                                            <?php else: ?>
                                                <span class="status-badge status-unpublished">Unpublished</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('M d, Y', strtotime($agenda['created_at'])); ?></td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem;">
                                                <a href="agenda-detail.php?id=<?php echo $agenda['id']; ?>" class="btn-sm btn-primary">View</a>
                                                <?php if (hasPermission('organizer.agendas.edit')): ?>
                                                <a href="edit-agenda.php?id=<?php echo $agenda['id']; ?>" class="btn-sm btn-secondary">Edit</a>
                                                <?php endif; ?>
                                                <?php if (hasPermission('organizer.agendas.delete')): ?>
                                                <button onclick="deleteAgenda(<?php echo $agenda['id']; ?>, '<?php echo htmlspecialchars(addslashes($agenda['title'])); ?>')" class="btn-sm" style="background: #dc2626; color: white; border: none; cursor: pointer;">Delete</button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script src="<?php echo $baseUrl; ?>/admin/assets/js/admin.js"></script>
    <script>
    function deleteAgenda(agendaId, agendaTitle) {
        if (!confirm('Are you sure you want to delete \"' + agendaTitle + '\"? This action cannot be undone.')) {
            return;
        }
        
        fetch('<?php echo $baseUrl; ?>/api/organizer/agendasapi.php?action=delete&id=' + agendaId, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Agenda deleted successfully');
                location.reload();
            } else {
                alert('Error: ' + (data.message || 'Failed to delete agenda'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while deleting the agenda');
        });
    }
    </script>
</body>
</html>


