<?php
/**
 * Organizer - Agenda Detail
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';
require_once __DIR__ . '/../../includes/url_helper.php';

requireOrganizer();

// Check permission to view agendas
if (!hasPermission('organizer.agendas.view')) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode('You do not have permission to view agendas.'));
    exit;
}

$agendaId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$agendaId) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/agendas.php?error=' . urlencode('Agenda ID is required.'));
    exit;
}

$organizer = getCurrentOrganizer();
$pdo = getDB();

// Soft-delete backward compatibility
try {
    $check = $pdo->query("SHOW COLUMNS FROM agendas LIKE 'deleted_at'");
    $hasDeletedAt = ($check && $check->rowCount() > 0);
} catch (Exception $e) {
    $hasDeletedAt = false;
}

// Verify ownership and fetch agenda
$sql = "SELECT * FROM agendas WHERE id = :id AND organizer_id = :organizer_id";
if ($hasDeletedAt) {
    $sql .= " AND deleted_at IS NULL";
}

$stmt = $pdo->prepare($sql);
$stmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
$agenda = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$agenda) {
    $baseUrl = getCorrectBaseUrl();
    header('Location: ' . $baseUrl . '/organizer/pages/agendas.php?error=' . urlencode('Agenda not found or you do not have permission to view this agenda.'));
    exit;
}

$baseUrl = getCorrectBaseUrl();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($agenda['title']); ?> | Organizer Panel</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo $baseUrl; ?>/admin/assets/css/admin.css">
</head>
<body class="organizer-panel">
    <?php 
    $current_page = 'agendas';
    include __DIR__ . '/../includes/header.php'; 
    ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/../includes/sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="page-header">
                <div>
                    <h1><?php echo htmlspecialchars($agenda['title']); ?></h1>
                    <p>Agenda Details</p>
                </div>
                <div>
                    <?php if (hasPermission('organizer.agendas.edit')): ?>
                    <a href="<?php echo $baseUrl; ?>/organizer/pages/edit-agenda.php?id=<?php echo $agenda['id']; ?>" class="btn btn-primary">Edit Agenda</a>
                    <?php endif; ?>
                    <a href="<?php echo $baseUrl; ?>/organizer/pages/agendas.php" class="btn btn-secondary">Back to Agendas</a>
                </div>
            </div>
            
            <div class="detail-container">
                <div class="detail-section">
                    <div class="detail-row">
                        <div class="detail-label">Title:</div>
                        <div class="detail-value"><strong><?php echo htmlspecialchars($agenda['title']); ?></strong></div>
                    </div>
                    
                    <div class="detail-row">
                        <div class="detail-label">Status:</div>
                        <div class="detail-value">
                            <?php if ($agenda['status'] === 'published'): ?>
                                <span class="badge badge-success">Published</span>
                            <?php else: ?>
                                <span class="badge badge-warning">Unpublished</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="detail-row">
                        <div class="detail-label">Time:</div>
                        <div class="detail-value">
                            <?php if ($agenda['start_time'] || $agenda['end_time']): ?>
                                <?php
                                    $start = $agenda['start_time'] ? date('F j, Y g:i A', strtotime($agenda['start_time'])) : '';
                                    $end = $agenda['end_time'] ? date('F j, Y g:i A', strtotime($agenda['end_time'])) : '';
                                ?>
                                <span><?php echo htmlspecialchars(trim($start . ($end ? ' - ' . $end : ''))); ?></span>
                            <?php else: ?>
                                <em style="color: #6b7280;">No time set</em>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if ($agenda['description']): ?>
                    <div class="detail-row">
                        <div class="detail-label">Description:</div>
                        <div class="detail-value"><?php echo nl2br(htmlspecialchars($agenda['description'])); ?></div>
                    </div>
                    <?php endif; ?>
                    
                    <div class="detail-row">
                        <div class="detail-label">Created:</div>
                        <div class="detail-value"><?php echo date('F j, Y g:i A', strtotime($agenda['created_at'])); ?></div>
                    </div>
                    
                    <?php if ($agenda['updated_at'] !== $agenda['created_at']): ?>
                    <div class="detail-row">
                        <div class="detail-label">Last Updated:</div>
                        <div class="detail-value"><?php echo date('F j, Y g:i A', strtotime($agenda['updated_at'])); ?></div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
    
    <style>
        .detail-container {
            background: white;
            border-radius: 8px;
            padding: 2rem;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .detail-section {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
        }
        .detail-row {
            display: grid;
            grid-template-columns: 150px 1fr;
            gap: 1rem;
            align-items: start;
        }
        .detail-label {
            font-weight: 600;
            color: #374151;
        }
        .detail-value {
            color: #111827;
        }
        @media (max-width: 768px) {
            .detail-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</body>
</html>


