<?php
/**
 * Organizer Login Page
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/roles.php';
require_once __DIR__ . '/../includes/permissions.php';

$error = $_GET['error'] ?? '';

// If already logged in as organizer, redirect to dashboard ONLY when user has some organizer permission.
// Otherwise show this page with the error (prevents infinite redirect loop).
if (isOrganizer() && empty($error)) {
    $hasAnyOrganizerPermission = hasAnyPermission([
        'organizer.events.view',
        'organizer.events.create',
        'organizer.events.edit',
        'organizer.events.delete',
        'organizer.attendees.view',
        'organizer.sales.view',
        'organizer.promotions.view',
        'organizer.refunds.view',
        'organizer.settings.view',
    ], $_SESSION['user_id'] ?? null);

    if ($hasAnyOrganizerPermission) {
        header('Location: ' . BASE_URL . '/organizer/dashboard.php');
        exit;
    } else {
        $error = 'No organizer permissions assigned. Please contact your administrator.';
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Organizer Login | beLIVE EventsHUB</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo BASE_URL; ?>/styles.css">
    <style>
        body {
            background: #f5f5f5;
            margin: 0;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
        }
        .organizer-login-wrapper {
            display: flex;
            min-height: 100vh;
        }
        .organizer-login-left {
            flex: 1;
            background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
            display: flex;
            flex-direction: column;
            justify-content: center;
            padding: 4rem 5rem;
            color: white;
            position: relative;
            overflow: hidden;
        }
        .organizer-login-left::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg width="100" height="100" xmlns="http://www.w3.org/2000/svg"><circle cx="50" cy="50" r="2" fill="rgba(255,255,255,0.1)"/></svg>');
            opacity: 0.3;
        }
        .organizer-login-left-content {
            position: relative;
            z-index: 1;
            max-width: 600px;
        }
        .organizer-logo-large {
            margin-bottom: 2rem;
        }
        .organizer-logo-large img {
            height: 60px;
            filter: brightness(0) invert(1);
        }
        .organizer-login-left h1 {
            font-size: 3.5rem;
            font-weight: 800;
            margin-bottom: 1rem;
            line-height: 1.1;
            letter-spacing: -0.03em;
        }
        .organizer-login-left h2 {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 1.5rem;
            opacity: 0.95;
        }
        .organizer-login-left p {
            font-size: 1.125rem;
            line-height: 1.8;
            margin-bottom: 2.5rem;
            opacity: 0.9;
        }
        .organizer-features {
            list-style: none;
            padding: 0;
            margin-top: 2rem;
        }
        .organizer-features li {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1.25rem;
            font-size: 1.05rem;
        }
        .organizer-features li::before {
            content: '✓';
            width: 32px;
            height: 32px;
            background: rgba(255, 255, 255, 0.2);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            flex-shrink: 0;
            font-size: 1.1rem;
        }
        .organizer-login-right {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 3rem;
            background: #fafafa;
        }
        .organizer-login-card {
            background: white;
            border-radius: 20px;
            padding: 3.5rem;
            max-width: 480px;
            width: 100%;
            box-shadow: 0 10px 40px rgba(37, 99, 235, 0.12);
            border: 1px solid rgba(37, 99, 235, 0.15);
        }
        .organizer-login-header {
            text-align: center;
            margin-bottom: 2.5rem;
        }
        .organizer-login-header h1 {
            font-size: 2rem;
            font-weight: 800;
            color: #2563eb;
            margin-bottom: 12px;
            letter-spacing: -0.02em;
        }
        .organizer-login-header p {
            font-size: 1rem;
            color: #6b7280;
        }
        .form-group {
            margin-bottom: 1.5rem;
        }
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            color: #374151;
            font-weight: 500;
            font-size: 0.875rem;
        }
        .form-group input {
            width: 100%;
            padding: 0.875rem;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 1rem;
            box-sizing: border-box;
            transition: border-color 0.2s;
        }
        .form-group input:focus {
            outline: none;
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }
        .btn-primary {
            width: 100%;
            padding: 0.875rem;
            background: #2563eb;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            background: #1e40af;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
        }
        .error-message {
            background: #fee2e2;
            color: #dc2626;
            padding: 0.875rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: <?php echo $error ? 'block' : 'none'; ?>;
            font-size: 0.875rem;
            border: 1px solid #fecaca;
        }
        .back-link {
            text-align: center;
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid #e5e7eb;
        }
        .back-link a {
            color: #6b7280;
            text-decoration: none;
            font-size: 0.875rem;
            transition: color 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        .back-link a:hover {
            color: #2563eb;
        }
        .organizer-credentials {
            background: #eff6ff;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1.5rem;
            font-size: 0.875rem;
        }
        .organizer-credentials strong {
            color: #1e40af;
            display: block;
            margin-bottom: 0.5rem;
        }
        .organizer-credentials code {
            background: white;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            color: #2563eb;
        }
        @media (max-width: 968px) {
            .organizer-login-wrapper {
                flex-direction: column;
            }
            .organizer-login-left {
                padding: 2.5rem;
                min-height: 50vh;
            }
            .organizer-login-left h1 {
                font-size: 2.5rem;
            }
            .organizer-login-left h2 {
                font-size: 1.25rem;
            }
            .organizer-login-right {
                padding: 2rem;
            }
            .organizer-login-card {
                padding: 2.5rem;
            }
        }
    </style>
</head>
<body class="organizer-panel">
    <div class="organizer-login-wrapper">
        <!-- Left Side - Branding & Features -->
        <div class="organizer-login-left">
            <div class="organizer-login-left-content">
                <div class="organizer-logo-large">
                    <img src="https://events.videostech.cloud/_next/image?url=%2Fbelive-logo.png&w=1920&q=75" alt="beLIVE">
                </div>
                <h1>Welcome to<br>Organizer Portal</h1>
                <h2>Manage your events, track sales, and grow your audience with powerful event management tools.</h2>
                <p>Create, promote, and manage your events all in one place.</p>
                
                <ul class="organizer-features">
                    <li>Complete event lifecycle management</li>
                    <li>Real-time sales and revenue tracking</li>
                    <li>Promotion and marketing tools</li>
                    <li>Attendee management and check-in</li>
                </ul>
            </div>
        </div>
        
        <!-- Right Side - Login Form -->
        <div class="organizer-login-right">
            <div class="organizer-login-card">
                <div class="organizer-login-header">
                    <h1>Organizer Login</h1>
                    <p>Sign in to access your organizer dashboard</p>
                </div>
                
                <div class="error-message" id="errorMessage" style="display: <?php echo $error ? 'block' : 'none'; ?>;">
                    <?php echo $error ? htmlspecialchars($error) : ''; ?>
                </div>
                
                <form id="organizerLoginForm" method="POST" action="<?php echo BASE_URL; ?>/api/organizer/loginapi.php">
                    <div class="form-group">
                        <label for="email">Email</label>
                        <input type="email" id="email" name="email" required autocomplete="email" value="organizer1@example.com">
                    </div>
                    <div class="form-group">
                        <label for="password">Password</label>
                        <input type="password" id="password" name="password" required autocomplete="current-password">
                    </div>
                    <button type="submit" class="btn-primary">Login</button>
                </form>
                
                <div class="organizer-credentials">
                    <strong>Test Organizer Credentials:</strong>
                    <div style="margin-top: 0.5rem;">
                        <strong>Email:</strong> <code>organizer1@example.com</code><br>
                        <strong>Password:</strong> <code>organizer123</code>
                    </div>
                </div>
                
                <div class="back-link">
                    <a href="<?php echo BASE_URL; ?>/index.html">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M19 12H5M12 19l-7-7 7-7"/>
                        </svg>
                        Back to Home
                    </a>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const loginForm = document.getElementById('organizerLoginForm');
        if (!loginForm) return;
        
        loginForm.addEventListener('submit', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const formData = new FormData(this);
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.textContent;
            const errorMsg = document.getElementById('errorMessage');
            
            if (errorMsg) {
                errorMsg.style.display = 'none';
            }
            
            submitBtn.disabled = true;
            submitBtn.textContent = 'Logging in...';
            
            fetch(this.action, {
                method: 'POST',
                body: formData
            })
            .then(async response => {
                // Log response for debugging
                console.log('Response status:', response.status);
                console.log('Response URL:', response.url);
                
                const contentType = response.headers.get('content-type');
                const isJson = contentType && contentType.includes('application/json');
                
                let data;
                if (isJson) {
                    data = await response.json();
                } else {
                    const text = await response.text();
                    console.error('Non-JSON response:', text);
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        throw new Error('Invalid response format: ' + text.substring(0, 100));
                    }
                }
                
                if (!response.ok) {
                    const errorMessage = data.message || data.errors?.join(', ') || 'Network response was not ok';
                    throw new Error(errorMessage);
                }
                
                return data;
            })
            .then(data => {
                console.log('Login response:', data);
                
                if (data && data.success) {
                    if (data.data && data.data.redirect) {
                        // Success - redirect
                        window.location.href = data.data.redirect;
                        return;
                    } else if (data.redirect) {
                        // Alternative redirect format
                        window.location.href = data.redirect;
                        return;
                    }
                }
                
                // Show error
                const errorMessage = data?.message || (data?.errors ? data.errors.join(', ') : null) || 'Login failed. Please try again.';
                if (errorMsg) {
                    errorMsg.textContent = errorMessage;
                    errorMsg.style.display = 'block';
                } else {
                    alert(errorMessage);
                }
                submitBtn.disabled = false;
                submitBtn.textContent = originalText;
            })
            .catch(error => {
                console.error('Login Error:', error);
                let errorMessage = 'An error occurred. Please try again.';
                
                // Try to get more details from the error
                if (error.message) {
                    errorMessage = error.message;
                }
                
                if (errorMsg) {
                    errorMsg.textContent = errorMessage + ' (Check console for details)';
                    errorMsg.style.display = 'block';
                } else {
                    alert(errorMessage);
                }
                submitBtn.disabled = false;
                submitBtn.textContent = originalText;
            });
            
            return false;
        });
    });
    </script>
</body>
</html>

