/**
 * Organizer Dashboard - Enhanced with Charts
 */

let performanceChart = null;
let statusChart = null;

document.addEventListener('DOMContentLoaded', function() {
    loadDashboardData();
});

function loadDashboardData() {
    fetch(ORGANIZER_API_BASE + 'dashboardapi.php', {
        method: 'GET',
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            populateDashboard(data.data);
        } else {
            showError('Failed to load dashboard data');
        }
    })
    .catch(error => {
        console.error('Dashboard API Error:', error);
        showError('Error loading dashboard data');
    });
}

function populateDashboard(data) {
    const stats = data.stats || {};
    const performanceData = data.performance_data || [];
    const statusDistribution = data.status_distribution || [];
    
    // Update summary cards
    updateSummaryCard('totalEvents', stats.total_events || 0);
    updateSummaryCard('publishedEvents', stats.published_events || 0);
    updateSummaryCard('draftEvents', stats.draft_events || 0);
    updateSummaryCard('cancelledEvents', stats.cancelled_events || 0);
    updateSummaryCard('totalArtists', stats.total_artists || 0);
    updateSummaryCard('totalSponsors', stats.total_sponsors || 0);
    
    // Update percentage changes (mock data for now)
    updatePercentageChange('totalEventsChange', 0.29);
    updatePercentageChange('publishedEventsChange', 0.29);
    updatePercentageChange('draftEventsChange', 0.18);
    updatePercentageChange('cancelledEventsChange', -0.12);
    updatePercentageChange('totalArtistsChange', 0.0);
    updatePercentageChange('totalSponsorsChange', 0.0);
    
    // Update tab counts
    updateTabCount('tabEventsCount', stats.total_events || 0);
    updateTabCount('tabArtistsCount', stats.total_artists || 0);
    updateTabCount('tabSponsorsCount', stats.total_sponsors || 0);
    updateTabCount('tabAgendasCount', stats.total_agendas || 0);
    
    // Initialize charts
    initPerformanceChart(performanceData);
    initStatusChart(statusDistribution);
}

function updateTabCount(elementId, value) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = numberFormat(value);
    }
}

function updatePercentageChange(elementId, percentage) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = Math.abs(percentage).toFixed(2) + '%';
    }
}

function updateSummaryCard(elementId, value) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = numberFormat(value);
    }
}

function initPerformanceChart(data) {
    const ctx = document.getElementById('performanceChart');
    if (!ctx) return;
    
    // Destroy existing chart if it exists
    if (performanceChart) {
        performanceChart.destroy();
    }
    
    const labels = data.map(item => item.date);
    const counts = data.map(item => item.count);
    
    performanceChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Events Created',
                data: counts,
                borderColor: '#dc2626',
                backgroundColor: 'rgba(220, 38, 38, 0.1)',
                tension: 0.4,
                fill: true,
                pointRadius: 4,
                pointHoverRadius: 6,
                pointBackgroundColor: '#dc2626',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: true,
                    position: 'top',
                    labels: {
                        usePointStyle: true,
                        padding: 15,
                        font: {
                            size: 12,
                            family: 'Inter'
                        }
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    padding: 12,
                    titleFont: {
                        size: 13,
                        family: 'Inter'
                    },
                    bodyFont: {
                        size: 12,
                        family: 'Inter'
                    },
                    cornerRadius: 8,
                    displayColors: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1,
                        font: {
                            size: 11,
                            family: 'Inter'
                        },
                        color: '#6b7280'
                    },
                    grid: {
                        color: '#f3f4f6'
                    }
                },
                x: {
                    ticks: {
                        font: {
                            size: 11,
                            family: 'Inter'
                        },
                        color: '#6b7280'
                    },
                    grid: {
                        display: false
                    }
                }
            }
        }
    });
}

function initStatusChart(data) {
    const ctx = document.getElementById('statusChart');
    if (!ctx) return;
    
    // Destroy existing chart if it exists
    if (statusChart) {
        statusChart.destroy();
    }
    
    // Filter out zero values for cleaner chart
    const filteredData = data.filter(item => item.value > 0);
    
    if (filteredData.length === 0) {
        ctx.parentElement.innerHTML = '<p style="text-align: center; color: #6b7280; padding: 2rem;">No data available</p>';
        return;
    }
    
    const labels = filteredData.map(item => item.label);
    const values = filteredData.map(item => item.value);
    const colors = filteredData.map(item => item.color);
    
    statusChart = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: labels,
            datasets: [{
                data: values,
                backgroundColor: colors,
                borderWidth: 0,
                hoverOffset: 8
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: true,
                    position: 'right',
                    labels: {
                        usePointStyle: true,
                        padding: 15,
                        font: {
                            size: 12,
                            family: 'Inter'
                        },
                        color: '#374151'
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    padding: 12,
                    titleFont: {
                        size: 13,
                        family: 'Inter'
                    },
                    bodyFont: {
                        size: 12,
                        family: 'Inter'
                    },
                    cornerRadius: 8,
                    callbacks: {
                        label: function(context) {
                            const label = context.label || '';
                            const value = context.parsed || 0;
                            const total = context.dataset.data.reduce((a, b) => a + b, 0);
                            const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : 0;
                            return label + ': ' + value + ' (' + percentage + '%)';
                        }
                    }
                }
            },
            cutout: '60%'
        }
    });
}

// Helper function for number formatting
function numberFormat(num) {
    return new Intl.NumberFormat('en-US').format(num);
}
