/**
 * Organizer Artists Management JavaScript
 */

const ARTISTS_API_BASE = BASE_URL + '/api/organizer/artistsapi.php';

let currentStatusFilter = 'all';
let currentSearch = '';

// Load artists on page load
document.addEventListener('DOMContentLoaded', function() {
    loadArtists();
    
    // Filter event listeners
    document.getElementById('applyFilters').addEventListener('click', function() {
        currentStatusFilter = document.getElementById('statusFilter').value;
        currentSearch = document.getElementById('searchInput').value.trim();
        loadArtists();
    });
    
    // Enter key on search input
    document.getElementById('searchInput').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            document.getElementById('applyFilters').click();
        }
    });
});

async function loadArtists() {
    const tbody = document.getElementById('artistsTableBody');
    if (!tbody) {
        console.error('Artists table body not found');
        return;
    }
    
    tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem;"><div class="loading-spinner"></div><p>Loading artists...</p></td></tr>';
    
    try {
        const params = new URLSearchParams();
        if (currentStatusFilter !== 'all') {
            params.append('status', currentStatusFilter);
        }
        if (currentSearch) {
            params.append('search', currentSearch);
        }
        
        const url = ARTISTS_API_BASE + (params.toString() ? '?' + params.toString() : '');
        const response = await fetch(url, {
            method: 'GET',
            credentials: 'same-origin'
        });
        
        const data = await response.json();
        
        if (!response.ok || !data.success) {
            throw new Error(data.message || 'Failed to load artists');
        }
        
        populateArtistsTable(data.data || []);
        
    } catch (error) {
        console.error('Error loading artists:', error);
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem; color: #dc2626;">Error loading artists: ' + escapeHtml(error.message) + '</td></tr>';
    }
}

function populateArtistsTable(artists) {
    const tbody = document.getElementById('artistsTableBody');
    if (!tbody) return;
    
    if (!artists || artists.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem;">No artists found.</td></tr>';
        return;
    }
    
    tbody.innerHTML = artists.map(artist => generateArtistRow(artist)).join('');
}

function generateArtistRow(artist) {
    const profileImage = artist.profile_image 
        ? `<img src="${escapeHtml(artist.profile_image)}" alt="${escapeHtml(artist.name)}" style="width: 50px; height: 50px; object-fit: cover; border-radius: 8px;">`
        : '<div style="width: 50px; height: 50px; background: #e5e7eb; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 20px;">🎤</div>';
    
    const statusBadge = artist.status === 'published' 
        ? '<span class="badge badge-success">Published</span>'
        : '<span class="badge badge-warning">Unpublished</span>';
    
    const bioPreview = artist.bio 
        ? (artist.bio.length > 100 ? escapeHtml(artist.bio.substring(0, 100)) + '...' : escapeHtml(artist.bio))
        : '<em style="color: #9ca3af;">No bio</em>';
    
    const createdDate = formatDate(artist.created_at);
    
    const editBtn = hasPermission('organizer.artists.edit')
        ? `<button onclick="editArtist(${artist.id})" class="btn btn-sm btn-secondary">Edit</button>`
        : '';
    
    const deleteBtn = hasPermission('organizer.artists.delete')
        ? `<button onclick="deleteArtist(${artist.id})" class="btn btn-sm btn-danger">Delete</button>`
        : '';
    
    const viewBtn = `<button onclick="viewArtist(${artist.id})" class="btn btn-sm btn-primary">View</button>`;
    
    return `
        <tr>
            <td>${profileImage}</td>
            <td><strong>${escapeHtml(artist.name)}</strong></td>
            <td>${bioPreview}</td>
            <td>${statusBadge}</td>
            <td>${createdDate}</td>
            <td>
                <div class="action-buttons">
                    ${viewBtn}
                    ${editBtn}
                    ${deleteBtn}
                </div>
            </td>
        </tr>
    `;
}

function viewArtist(id) {
    window.location.href = BASE_URL + '/organizer/pages/artist-detail.php?id=' + id;
}

function editArtist(id) {
    window.location.href = BASE_URL + '/organizer/pages/edit-artist.php?id=' + id;
}

async function deleteArtist(id) {
    if (!confirm('Are you sure you want to delete this artist? This action cannot be undone.')) {
        return;
    }
    
    try {
        const response = await fetch(ARTISTS_API_BASE + '?action=delete&id=' + id, {
            method: 'DELETE',
            credentials: 'same-origin'
        });
        
        const data = await response.json();
        
        if (!response.ok || !data.success) {
            throw new Error(data.message || 'Failed to delete artist');
        }
        
        showSuccess('Artist deleted successfully');
        loadArtists();
        
    } catch (error) {
        console.error('Error deleting artist:', error);
        showError('Failed to delete artist: ' + error.message);
    }
}

// Helper functions (use existing or define)
function hasPermission(permission) {
    // This should be set by PHP or checked via API
    // For now, assume true if buttons are rendered
    return true;
}

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
}

function showSuccess(message) {
    if (typeof window.showSuccess === 'function') {
        window.showSuccess(message);
    } else {
        alert(message);
    }
}

function showError(message) {
    if (typeof window.showError === 'function') {
        window.showError(message);
    } else {
        alert(message);
    }
}

