/**
 * Organizer Panel - Shared API Functions
 */

// Base API URL
const ORGANIZER_API_BASE = BASE_URL + '/api/organizer/';

/**
 * Generic API fetch function
 */
async function fetchOrganizerAPI(endpoint, options = {}) {
    const url = ORGANIZER_API_BASE + endpoint;
    const defaultOptions = {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/json'
        }
    };
    
    const config = { ...defaultOptions, ...options };
    
    try {
        const response = await fetch(url, config);
        const data = await response.json();
        
        if (!response.ok) {
            throw new Error(data.message || 'API request failed');
        }
        
        return data;
    } catch (error) {
        console.error('API Error:', error);
        throw error;
    }
}

// Reuse admin helper functions
if (typeof numberFormat === 'undefined') {
    function numberFormat(num, decimals = 0) {
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(num);
    }
}

if (typeof formatDate === 'undefined') {
    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
    }
}

if (typeof formatCurrency === 'undefined') {
    function formatCurrency(amount) {
        return '$' + numberFormat(amount, 2);
    }
}

if (typeof capitalizeFirst === 'undefined') {
    function capitalizeFirst(str) {
        if (!str) return '';
        return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
    }
}

if (typeof escapeHtml === 'undefined') {
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

if (typeof showError === 'undefined') {
    function showError(message, container = '.admin-content') {
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.style.cssText = 'background: #fee2e2; color: #dc2626; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';
        errorDiv.textContent = message;
        const content = document.querySelector(container);
        if (content) {
            content.insertBefore(errorDiv, content.firstChild);
            setTimeout(() => errorDiv.remove(), 5000);
        }
    }
}

if (typeof showLoading === 'undefined') {
    function showLoading(elementId) {
        const element = document.getElementById(elementId);
        if (element) {
            element.style.display = 'block';
            element.innerHTML = '<p style="text-align: center; padding: 2rem;">Loading...</p>';
        }
    }
}

if (typeof hideLoading === 'undefined') {
    function hideLoading(elementId) {
        const element = document.getElementById(elementId);
        if (element) {
            element.style.display = 'none';
        }
    }
}

if (typeof buildQueryString === 'undefined') {
    function buildQueryString(params) {
        const query = new URLSearchParams();
        Object.keys(params).forEach(key => {
            if (params[key] !== null && params[key] !== undefined && params[key] !== '') {
                query.append(key, params[key]);
            }
        });
        return query.toString();
    }
}

