-- Role and Permission System Migration
-- This migration adds tables for flexible role and permission management

-- Roles Table
CREATE TABLE IF NOT EXISTS `roles` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(100) NOT NULL,
  `slug` VARCHAR(100) NOT NULL,
  `description` TEXT DEFAULT NULL,
  `is_system_role` TINYINT(1) DEFAULT 0 COMMENT 'System roles (super_admin, admin) cannot be deleted',
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `slug` (`slug`),
  KEY `is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Permissions Table
CREATE TABLE IF NOT EXISTS `permissions` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(100) NOT NULL,
  `slug` VARCHAR(100) NOT NULL,
  `module` VARCHAR(50) NOT NULL COMMENT 'Module name (events, users, organizers, etc.)',
  `description` TEXT DEFAULT NULL,
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `slug` (`slug`),
  KEY `module` (`module`),
  KEY `is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Role Permissions Junction Table
CREATE TABLE IF NOT EXISTS `role_permissions` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `role_id` INT(11) NOT NULL,
  `permission_id` INT(11) NOT NULL,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `role_permission` (`role_id`, `permission_id`),
  KEY `role_id` (`role_id`),
  KEY `permission_id` (`permission_id`),
  FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`permission_id`) REFERENCES `permissions`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- User Roles Junction Table (for assigning roles to all user types: admin, organizer, venue)
CREATE TABLE IF NOT EXISTS `user_roles` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) NOT NULL COMMENT 'References customers.id',
  `role_id` INT(11) NOT NULL,
  `user_type` ENUM('admin', 'organizer', 'venue', 'customer') DEFAULT 'admin' COMMENT 'Type of user this role is assigned to',
  `assigned_by` INT(11) DEFAULT NULL COMMENT 'Admin who assigned this role',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `user_role_type` (`user_id`, `role_id`, `user_type`),
  KEY `user_id` (`user_id`),
  KEY `role_id` (`role_id`),
  KEY `user_type` (`user_type`),
  KEY `assigned_by` (`assigned_by`),
  FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`user_id`) REFERENCES `customers`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`assigned_by`) REFERENCES `customers`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default system roles for ADMIN
INSERT INTO `roles` (`name`, `slug`, `description`, `is_system_role`, `is_active`) VALUES
('Super Admin', 'super_admin', 'Full access to all features and settings', 1, 1),
('Admin', 'admin', 'Full access to all features except system settings', 1, 1)
ON DUPLICATE KEY UPDATE `name` = VALUES(`name`);

-- Insert default roles for ORGANIZERS
INSERT INTO `roles` (`name`, `slug`, `description`, `is_system_role`, `is_active`) VALUES
('Organizer Manager', 'organizer_manager', 'Full access to organizer features', 1, 1),
('Organizer Staff', 'organizer_staff', 'Limited access to organizer features', 1, 1)
ON DUPLICATE KEY UPDATE `name` = VALUES(`name`);

-- Insert default roles for VENUES
INSERT INTO `roles` (`name`, `slug`, `description`, `is_system_role`, `is_active`) VALUES
('Venue Manager', 'venue_manager', 'Full access to venue features', 1, 1),
('Venue Staff', 'venue_staff', 'Limited access to venue features', 1, 1)
ON DUPLICATE KEY UPDATE `name` = VALUES(`name`);

-- Insert default permissions for ADMIN
INSERT INTO `permissions` (`name`, `slug`, `module`, `description`, `is_active`) VALUES
-- Dashboard
('View Dashboard', 'dashboard.view', 'dashboard', 'View admin dashboard', 1),

-- Users Management
('View Users', 'users.view', 'users', 'View users list', 1),
('Create Users', 'users.create', 'users', 'Create new users', 1),
('Edit Users', 'users.edit', 'users', 'Edit user details', 1),
('Delete Users', 'users.delete', 'users', 'Delete users', 1),

-- Organizers Management
('View Organizers', 'organizers.view', 'organizers', 'View organizers list', 1),
('Create Organizers', 'organizers.create', 'organizers', 'Create new organizers', 1),
('Edit Organizers', 'organizers.edit', 'organizers', 'Edit organizer details', 1),
('Delete Organizers', 'organizers.delete', 'organizers', 'Delete organizers', 1),
('Approve Organizers', 'organizers.approve', 'organizers', 'Approve organizer accounts', 1),

-- Venues Management
('View Venues', 'venues.view', 'venues', 'View venues list', 1),
('Create Venues', 'venues.create', 'venues', 'Create new venues', 1),
('Edit Venues', 'venues.edit', 'venues', 'Edit venue details', 1),
('Delete Venues', 'venues.delete', 'venues', 'Delete venues', 1),
('Approve Venues', 'venues.approve', 'venues', 'Approve venue accounts', 1),

-- Events Management
('View Events', 'events.view', 'events', 'View events list', 1),
('Create Events', 'events.create', 'events', 'Create new events', 1),
('Edit Events', 'events.edit', 'events', 'Edit event details', 1),
('Delete Events', 'events.delete', 'events', 'Delete events', 1),
('Approve Events', 'events.approve', 'events', 'Approve events', 1),
('Reject Events', 'events.reject', 'events', 'Reject events', 1),

-- Analytics
('View Analytics', 'analytics.view', 'analytics', 'View analytics and reports', 1),
('Export Analytics', 'analytics.export', 'analytics', 'Export analytics data', 1),

-- Wallet Management
('View Wallet', 'wallet.view', 'wallet', 'View wallet transactions', 1),
('Manage Wallet', 'wallet.manage', 'wallet', 'Manage wallet settings', 1),

-- Settlements
('View Settlements', 'settlements.view', 'settlements', 'View settlements', 1),
('Process Settlements', 'settlements.process', 'settlements', 'Process settlements', 1),
('Approve Settlements', 'settlements.approve', 'settlements', 'Approve settlements', 1),

-- Loyalty Program
('View Loyalty', 'loyalty.view', 'loyalty', 'View loyalty program', 1),
('Manage Loyalty', 'loyalty.manage', 'loyalty', 'Manage loyalty program', 1),

-- CMS
('View CMS', 'cms.view', 'cms', 'View CMS content', 1),
('Edit CMS', 'cms.edit', 'cms', 'Edit CMS content', 1),

-- Notifications
('View Notifications', 'notifications.view', 'notifications', 'View notifications', 1),
('Send Notifications', 'notifications.send', 'notifications', 'Send notifications', 1),

-- Roles & Permissions
('View Roles', 'roles.view', 'roles', 'View roles list', 1),
('Create Roles', 'roles.create', 'roles', 'Create new roles', 1),
('Edit Roles', 'roles.edit', 'roles', 'Edit roles', 1),
('Delete Roles', 'roles.delete', 'roles', 'Delete roles', 1),
('Manage Permissions', 'roles.permissions', 'roles', 'Manage role permissions', 1),

-- Settings
('View Settings', 'settings.view', 'settings', 'View system settings', 1),
('Edit Settings', 'settings.edit', 'settings', 'Edit system settings', 1)
ON DUPLICATE KEY UPDATE `name` = VALUES(`name`);

-- Insert permissions for ORGANIZERS
INSERT INTO `permissions` (`name`, `slug`, `module`, `description`, `is_active`) VALUES
-- Organizer Events
('View My Events', 'organizer.events.view', 'organizer_events', 'View own events', 1),
('Create Events', 'organizer.events.create', 'organizer_events', 'Create new events', 1),
('Edit My Events', 'organizer.events.edit', 'organizer_events', 'Edit own events', 1),
('Delete My Events', 'organizer.events.delete', 'organizer_events', 'Delete own events', 1),
('Publish Events', 'organizer.events.publish', 'organizer_events', 'Publish events', 1),

-- Organizer Attendees
('View Attendees', 'organizer.attendees.view', 'organizer_attendees', 'View event attendees', 1),
('Manage Attendees', 'organizer.attendees.manage', 'organizer_attendees', 'Manage attendees', 1),

-- Organizer Sales
('View Sales', 'organizer.sales.view', 'organizer_sales', 'View sales reports', 1),
('Export Sales', 'organizer.sales.export', 'organizer_sales', 'Export sales data', 1),

-- Organizer Promotions
('View Promotions', 'organizer.promotions.view', 'organizer_promotions', 'View promotions', 1),
('Create Promotions', 'organizer.promotions.create', 'organizer_promotions', 'Create promotions', 1),
('Edit Promotions', 'organizer.promotions.edit', 'organizer_promotions', 'Edit promotions', 1),
('Delete Promotions', 'organizer.promotions.delete', 'organizer_promotions', 'Delete promotions', 1),

-- Organizer Refunds
('View Refunds', 'organizer.refunds.view', 'organizer_refunds', 'View refund requests', 1),
('Process Refunds', 'organizer.refunds.process', 'organizer_refunds', 'Process refunds', 1),

-- Organizer Settings
('View Organizer Settings', 'organizer.settings.view', 'organizer_settings', 'View organizer settings', 1),
('Edit Organizer Settings', 'organizer.settings.edit', 'organizer_settings', 'Edit organizer settings', 1)
ON DUPLICATE KEY UPDATE `name` = VALUES(`name`);

-- Insert permissions for VENUES
INSERT INTO `permissions` (`name`, `slug`, `module`, `description`, `is_active`) VALUES
-- Venue Events
('View Venue Events', 'venue.events.view', 'venue_events', 'View events at venue', 1),
('Manage Venue Events', 'venue.events.manage', 'venue_events', 'Manage venue events', 1),

-- Venue Access Control
('View Access Control', 'venue.access.view', 'venue_access', 'View access control', 1),
('Manage Access Control', 'venue.access.manage', 'venue_access', 'Manage access control', 1),
('Manage Staff', 'venue.staff.manage', 'venue_access', 'Manage venue staff', 1),

-- Venue Analytics
('View Venue Analytics', 'venue.analytics.view', 'venue_analytics', 'View venue analytics', 1),
('Export Analytics', 'venue.analytics.export', 'venue_analytics', 'Export analytics', 1),

-- Venue Profile
('View Venue Profile', 'venue.profile.view', 'venue_profile', 'View venue profile', 1),
('Edit Venue Profile', 'venue.profile.edit', 'venue_profile', 'Edit venue profile', 1),

-- Venue Settings
('View Venue Settings', 'venue.settings.view', 'venue_settings', 'View venue settings', 1),
('Edit Venue Settings', 'venue.settings.edit', 'venue_settings', 'Edit venue settings', 1)
ON DUPLICATE KEY UPDATE `name` = VALUES(`name`);

-- Assign all permissions to super_admin role
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id
FROM `roles` r
CROSS JOIN `permissions` p
WHERE r.slug = 'super_admin'
ON DUPLICATE KEY UPDATE `role_id` = VALUES(`role_id`);

-- Assign all permissions to admin role (except system settings)
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id
FROM `roles` r
CROSS JOIN `permissions` p
WHERE r.slug = 'admin' AND p.slug NOT LIKE 'settings.%' AND p.module NOT LIKE 'organizer_%' AND p.module NOT LIKE 'venue_%'
ON DUPLICATE KEY UPDATE `role_id` = VALUES(`role_id`);

-- Assign all organizer permissions to organizer_manager role
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id
FROM `roles` r
CROSS JOIN `permissions` p
WHERE r.slug = 'organizer_manager' AND p.module LIKE 'organizer_%'
ON DUPLICATE KEY UPDATE `role_id` = VALUES(`role_id`);

-- Assign limited permissions to organizer_staff role
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id
FROM `roles` r
CROSS JOIN `permissions` p
WHERE r.slug = 'organizer_staff' 
AND p.slug IN (
    'organizer.events.view',
    'organizer.attendees.view',
    'organizer.sales.view',
    'organizer.promotions.view',
    'organizer.refunds.view',
    'organizer.settings.view'
)
ON DUPLICATE KEY UPDATE `role_id` = VALUES(`role_id`);

-- Assign all venue permissions to venue_manager role
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id
FROM `roles` r
CROSS JOIN `permissions` p
WHERE r.slug = 'venue_manager' AND p.module LIKE 'venue_%'
ON DUPLICATE KEY UPDATE `role_id` = VALUES(`role_id`);

-- Assign limited permissions to venue_staff role
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id
FROM `roles` r
CROSS JOIN `permissions` p
WHERE r.slug = 'venue_staff' 
AND p.slug IN (
    'venue.events.view',
    'venue.access.view',
    'venue.analytics.view',
    'venue.profile.view',
    'venue.settings.view'
)
ON DUPLICATE KEY UPDATE `role_id` = VALUES(`role_id`);

-- Update existing admins table to link with roles
-- First, ensure all existing admins have super_admin or admin role assigned
-- This will be handled by PHP migration script

