<?php
/**
 * User Permissions Helper - Works directly with customers table
 * This file provides helper functions to work with customers table and roles/permissions
 */

require_once __DIR__ . '/permissions.php';

/**
 * Get user info with roles and permissions (from customers table)
 */
function getUserWithRoles($userId = null) {
    if (!isLoggedIn()) {
        return null;
    }
    
    $userId = $userId ?? $_SESSION['user_id'];
    
    try {
        $pdo = getDB();
        
        // Get user from customers table
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = :id AND is_active = 1");
        $stmt->execute([':id' => $userId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return null;
        }
        
        // Get user type
        $userType = getCurrentUserType($userId);
        $user['user_type'] = $userType;
        
        // Get roles
        $user['roles'] = getUserRoles($userId, $userType);
        
        // Get permissions
        $user['permissions'] = getUserPermissions($userId, $userType);
        
        // Get additional info based on user type
        if ($userType === 'admin') {
            $stmt = $pdo->prepare("
                SELECT a.* FROM admins a
                WHERE a.user_id = :user_id AND a.is_active = 1
            ");
            $stmt->execute([':user_id' => $userId]);
            $adminInfo = $stmt->fetch();
            if ($adminInfo) {
                $user['admin_info'] = $adminInfo;
            }
        } elseif ($userType === 'organizer') {
            $stmt = $pdo->prepare("
                SELECT o.* FROM organizers o
                WHERE o.user_id = :user_id AND o.is_active = 1
            ");
            $stmt->execute([':user_id' => $userId]);
            $organizerInfo = $stmt->fetch();
            if ($organizerInfo) {
                $user['organizer_info'] = $organizerInfo;
            }
        } elseif ($userType === 'venue') {
            $stmt = $pdo->prepare("
                SELECT v.* FROM venues v
                WHERE v.user_id = :user_id AND v.is_active = 1
            ");
            $stmt->execute([':user_id' => $userId]);
            $venueInfo = $stmt->fetch();
            if ($venueInfo) {
                $user['venue_info'] = $venueInfo;
            }
        }
        
        return $user;
    } catch (Exception $e) {
        error_log('Error getting user with roles: ' . $e->getMessage());
        return null;
    }
}

/**
 * Check if customer (from customers table) has permission
 * This is the main function to use - works with customers table directly
 */
function customerHasPermission($permissionSlug, $customerId = null) {
    return hasPermission($permissionSlug, $customerId);
}

/**
 * Assign role to customer (works with customers table)
 */
function assignRoleToCustomer($customerId, $roleId, $userType = null, $assignedBy = null) {
    try {
        $pdo = getDB();
        
        // Auto-detect user type if not provided
        if (!$userType) {
            $userType = getCurrentUserType($customerId);
        }
        
        // Check if customer exists
        $stmt = $pdo->prepare("SELECT id FROM customers WHERE id = :id AND is_active = 1");
        $stmt->execute([':id' => $customerId]);
        if (!$stmt->fetch()) {
            return ['success' => false, 'message' => 'Customer not found'];
        }
        
        // Check if role exists
        $role = getRoleById($roleId);
        if (!$role) {
            return ['success' => false, 'message' => 'Role not found'];
        }
        
        // Check if already assigned
        $stmt = $pdo->prepare("
            SELECT id FROM user_roles 
            WHERE user_id = :user_id AND role_id = :role_id AND user_type = :user_type
        ");
        $stmt->execute([
            ':user_id' => $customerId,
            ':role_id' => $roleId,
            ':user_type' => $userType
        ]);
        if ($stmt->fetch()) {
            return ['success' => false, 'message' => 'Role already assigned'];
        }
        
        // Assign role
        $assignedBy = $assignedBy ?? $_SESSION['user_id'] ?? $customerId;
        $stmt = $pdo->prepare("
            INSERT INTO user_roles (user_id, role_id, user_type, assigned_by)
            VALUES (:user_id, :role_id, :user_type, :assigned_by)
        ");
        $stmt->execute([
            ':user_id' => $customerId,
            ':role_id' => $roleId,
            ':user_type' => $userType,
            ':assigned_by' => $assignedBy
        ]);
        
        return ['success' => true, 'message' => 'Role assigned successfully'];
    } catch (Exception $e) {
        error_log('Error assigning role: ' . $e->getMessage());
        return ['success' => false, 'message' => 'Error assigning role: ' . $e->getMessage()];
    }
}

/**
 * Remove role from customer (works with customers table)
 */
function removeRoleFromCustomer($customerId, $roleId, $userType = null) {
    try {
        $pdo = getDB();
        
        // Auto-detect user type if not provided
        if (!$userType) {
            $userType = getCurrentUserType($customerId);
        }
        
        // Remove role
        $stmt = $pdo->prepare("
            DELETE FROM user_roles 
            WHERE user_id = :user_id AND role_id = :role_id AND user_type = :user_type
        ");
        $stmt->execute([
            ':user_id' => $customerId,
            ':role_id' => $roleId,
            ':user_type' => $userType
        ]);
        
        return ['success' => true, 'message' => 'Role removed successfully'];
    } catch (Exception $e) {
        error_log('Error removing role: ' . $e->getMessage());
        return ['success' => false, 'message' => 'Error removing role: ' . $e->getMessage()];
    }
}

/**
 * Get all customers with their roles and permissions
 */
function getAllCustomersWithRoles($userType = null) {
    try {
        $pdo = getDB();
        
        $sql = "SELECT c.* FROM customers c WHERE c.is_active = 1";
        if ($userType) {
            // Filter by user type
            if ($userType === 'admin') {
                $sql .= " AND EXISTS (SELECT 1 FROM admins a WHERE a.user_id = c.id AND a.is_active = 1)";
            } elseif ($userType === 'organizer') {
                $sql .= " AND EXISTS (SELECT 1 FROM organizers o WHERE o.user_id = c.id AND o.is_active = 1)";
            } elseif ($userType === 'venue') {
                $sql .= " AND EXISTS (SELECT 1 FROM venues v WHERE v.user_id = c.id AND v.is_active = 1)";
            }
        }
        $sql .= " ORDER BY c.created_at DESC";
        
        $stmt = $pdo->query($sql);
        $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Add roles and permissions to each customer
        foreach ($customers as &$customer) {
            $customerType = getCurrentUserType($customer['id']);
            $customer['user_type'] = $customerType;
            $customer['roles'] = getUserRoles($customer['id'], $customerType);
            $customer['permissions'] = getUserPermissions($customer['id'], $customerType);
        }
        
        return $customers;
    } catch (Exception $e) {
        error_log('Error getting customers with roles: ' . $e->getMessage());
        return [];
    }
}



