<?php
/**
 * Role-based authentication and authorization
 */

require_once __DIR__ . '/auth.php';

/**
 * Check if user has admin role
 */
function isAdmin() {
    if (!isLoggedIn()) {
        return false;
    }
    
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            SELECT a.* FROM admins a
            INNER JOIN customers c ON a.user_id = c.id
            WHERE a.user_id = :user_id AND a.is_active = 1 AND c.is_active = 1
        ");
        $stmt->execute([':user_id' => $_SESSION['user_id']]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log('Error checking admin role: ' . $e->getMessage());
        return false;
    }
}

/**
 * Check if user is organizer
 */
function isOrganizer() {
    if (!isLoggedIn()) {
        return false;
    }
    
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            SELECT o.* FROM organizers o
            INNER JOIN customers c ON o.user_id = c.id
            WHERE o.user_id = :user_id AND o.is_active = 1 AND c.is_active = 1
        ");
        $stmt->execute([':user_id' => $_SESSION['user_id']]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log('Error checking organizer role: ' . $e->getMessage());
        return false;
    }
}

/**
 * Check if user is venue
 */
function isVenue() {
    if (!isLoggedIn()) {
        return false;
    }
    
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            SELECT v.* FROM venues v
            INNER JOIN customers c ON v.user_id = c.id
            WHERE v.user_id = :user_id AND v.is_active = 1 AND c.is_active = 1
        ");
        $stmt->execute([':user_id' => $_SESSION['user_id']]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log('Error checking venue role: ' . $e->getMessage());
        return false;
    }
}

/**
 * Get user role type
 */
function getUserRole() {
    if (!isLoggedIn()) {
        return null;
    }
    
    if (isAdmin()) {
        return 'admin';
    } elseif (isOrganizer()) {
        return 'organizer';
    } elseif (isVenue()) {
        return 'venue';
    }
    
    return 'customer';
}

/**
 * Get base URL for redirects - Uses BASE_URL constant from config.php
 */
function getBaseUrlForRedirect() {
    // Always use BASE_URL constant if defined (single source of truth)
    if (defined('BASE_URL')) {
        return BASE_URL;
    }
    
    // Fallback: Use getCorrectBaseUrl() helper
    if (function_exists('getCorrectBaseUrl')) {
        return getCorrectBaseUrl();
    }
    
    // Last resort fallback
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    if (strpos($host, 'belive-events.videostech.cloud') !== false) {
        return $protocol . '://' . $host;
    }
    
    return $protocol . '://' . $host . '/events_belive';
}

/**
 * Require admin access
 */
function requireAdmin() {
    // Check if this is an API request (JSON response expected)
    $isApiRequest = (
        strpos($_SERVER['REQUEST_URI'] ?? '', '/api/') !== false ||
        (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)
    );
    
    if (!isLoggedIn()) {
        if ($isApiRequest) {
            http_response_code(401);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Authentication required', 'error' => 'Please login first']);
            exit;
        }
        $baseUrl = getBaseUrlForRedirect();
        header('Location: ' . $baseUrl . '/admin/login.php?error=' . urlencode('Please login first'));
        exit;
    }
    
    if (!isAdmin()) {
        if ($isApiRequest) {
            http_response_code(403);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Admin access required', 'error' => 'Insufficient permissions']);
            exit;
        }
        $baseUrl = getBaseUrlForRedirect();
        header('Location: ' . $baseUrl . '/admin/login.php?error=' . urlencode('Admin access required'));
        exit;
    }
}

/**
 * Require organizer access
 */
function requireOrganizer() {
    if (!isLoggedIn()) {
        $baseUrl = getBaseUrlForRedirect();
        header('Location: ' . $baseUrl . '/organizer/login.php?error=' . urlencode('Please login first'));
        exit;
    }
    
    if (!isOrganizer()) {
        $baseUrl = getBaseUrlForRedirect();
        header('Location: ' . $baseUrl . '/organizer/login.php?error=' . urlencode('Organizer access required'));
        exit;
    }
}

/**
 * Require venue access
 */
function requireVenue() {
    // Check if this is an API request (JSON response expected)
    $isApiRequest = (
        strpos($_SERVER['REQUEST_URI'] ?? '', '/api/') !== false ||
        (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)
    );
    
    if (!isLoggedIn()) {
        if ($isApiRequest) {
            http_response_code(401);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Authentication required', 'error' => 'Please login first']);
            exit;
        }
        $baseUrl = getBaseUrlForRedirect();
        header('Location: ' . $baseUrl . '/venue/login.php?error=' . urlencode('Please login first'));
        exit;
    }
    
    if (!isVenue()) {
        if ($isApiRequest) {
            http_response_code(403);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Venue access required', 'error' => 'Insufficient permissions']);
            exit;
        }
        $baseUrl = getBaseUrlForRedirect();
        header('Location: ' . $baseUrl . '/venue/login.php?error=' . urlencode('Venue access required'));
        exit;
    }
}

/**
 * Get current admin data
 */
function getCurrentAdmin() {
    return isAdmin();
}

/**
 * Get current organizer data
 */
function getCurrentOrganizer() {
    return isOrganizer();
}

/**
 * Get current venue data
 */
function getCurrentVenue() {
    return isVenue();
}

