<?php
/**
 * Event Recommendation System
 * Provides personalized event recommendations based on user activity and preferences
 */

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/user_activity.php';

/**
 * Get recommended events for a user
 * Returns events sorted by relevance score
 */
function getRecommendedEvents($customerId, $limit = 20, $eventType = null, $country = null) {
    try {
        $pdo = getDB();
        
        // Check if user_preferences table exists
        try {
            $pdo->query("SELECT 1 FROM user_preferences LIMIT 1");
        } catch (Exception $e) {
            // Table doesn't exist, return all events
            return getAllEvents($limit, $eventType, $country);
        }
        
        // Get user preferences
        $preferences = getUserPreferences($customerId);
        
        // If no preferences, return all events (no filtering)
        if (empty($preferences)) {
            return getAllEvents($limit, $eventType, $country);
        }
        
        // Build recommendation query with scoring
        $sql = "
            SELECT 
                e.*,
                (
                    -- Event type match
                    CASE 
                        WHEN e.event_type IN (
                            SELECT preference_value 
                            FROM user_preferences 
                            WHERE customer_id = :customer_id 
                            AND preference_type = 'event_type'
                        ) THEN 10
                        ELSE 0
                    END +
                    -- Location match
                    CASE 
                        WHEN e.city IN (
                            SELECT preference_value 
                            FROM user_preferences 
                            WHERE customer_id = :customer_id 
                            AND preference_type = 'location'
                        ) OR e.country IN (
                            SELECT preference_value 
                            FROM user_preferences 
                            WHERE customer_id = :customer_id 
                            AND preference_type = 'location'
                        ) THEN 8
                        ELSE 0
                    END +
                    -- Keyword match in name/description
                    (
                        SELECT SUM(up.weight * 2)
                        FROM user_preferences up
                        WHERE up.customer_id = :customer_id
                        AND (
                            e.name LIKE CONCAT('%', up.preference_value, '%')
                            OR e.description LIKE CONCAT('%', up.preference_value, '%')
                            OR e.attendee_profiles LIKE CONCAT('%', up.preference_value, '%')
                        )
                    ) +
                    -- Interest keyword match
                    (
                        SELECT COUNT(*) * 1.5
                        FROM customers c
                        LEFT JOIN user_activity ua ON c.id = ua.customer_id
                        WHERE c.id = :customer_id
                        AND ua.keywords IS NOT NULL
                        AND (
                            e.name LIKE CONCAT('%', ua.keywords, '%')
                            OR e.description LIKE CONCAT('%', ua.keywords, '%')
                            OR e.event_type LIKE CONCAT('%', ua.keywords, '%')
                        )
                    )
                ) as relevance_score
            FROM events e
            WHERE 1=1
        ";
        
        $params = [':customer_id' => $customerId];
        
        // Apply filters
        if ($eventType && $eventType !== 'all') {
            $sql .= " AND e.event_type = :event_type";
            $params[':event_type'] = $eventType;
        }
        
        if ($country) {
            $sql .= " AND e.country = :country";
            $params[':country'] = $country;
        }
        
        $sql .= " ORDER BY relevance_score DESC, e.start_date ASC, e.name ASC LIMIT :limit";
        
        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->execute();
        
        $events = $stmt->fetchAll();
        
        // If we got fewer events than requested, fill with regular events
        if (count($events) < $limit) {
            $remaining = $limit - count($events);
            $regularEvents = getAllEvents($remaining, $eventType, $country, array_column($events, 'id'));
            $events = array_merge($events, $regularEvents);
        }
        
        return $events;
        
    } catch (Exception $e) {
        error_log('Error getting recommended events: ' . $e->getMessage());
        // Fallback to all events
        return getAllEvents($limit, $eventType, $country);
    }
}

/**
 * Get all events (fallback when no preferences)
 */
function getAllEvents($limit = 20, $eventType = null, $country = null, $excludeIds = []) {
    try {
        $pdo = getDB();
        
        $sql = "SELECT * FROM events WHERE 1=1";
        $params = [];
        
        if ($eventType && $eventType !== 'all') {
            $sql .= " AND event_type = :event_type";
            $params[':event_type'] = $eventType;
        }
        
        if ($country) {
            $sql .= " AND country = :country";
            $params[':country'] = $country;
        }
        
        if (!empty($excludeIds)) {
            $placeholders = [];
            $excludeIndex = 0;
            foreach ($excludeIds as $excludeId) {
                $paramName = ':exclude_' . $excludeIndex++;
                $placeholders[] = $paramName;
                $params[$paramName] = $excludeId;
            }
            $sql .= " AND id NOT IN (" . implode(',', $placeholders) . ")";
        }
        
        $sql .= " ORDER BY start_date ASC, name ASC LIMIT :limit";
        
        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
        
    } catch (Exception $e) {
        error_log('Error getting all events: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get personalized events for logged-in user
 * Falls back to all events if user not logged in or no preferences
 */
function getPersonalizedEvents($eventType = null, $country = null, $limit = 50) {
    // Check if user is logged in
    if (!isLoggedIn()) {
        return getAllEvents($limit, $eventType, $country);
    }
    
    $customerId = $_SESSION['user_id'];
    
    // Get recommended events
    $events = getRecommendedEvents($customerId, $limit, $eventType, $country);
    
    // Always ensure we have events (fallback)
    if (empty($events)) {
        $events = getAllEvents($limit, $eventType, $country);
    }
    
    return $events;
}
