<?php
/**
 * Role and Permission Management System
 */

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/roles.php';

/**
 * Get current user type (admin, organizer, venue, customer)
 * Automatically detects from customers table and related tables
 */
function getCurrentUserType($userId = null) {
    if (!isLoggedIn()) {
        return null;
    }
    
    $userId = $userId ?? $_SESSION['user_id'];
    
    try {
        $pdo = getDB();
        
        // Check in order: admin -> organizer -> venue -> customer
        // Admin check
        $stmt = $pdo->prepare("
            SELECT 'admin' as user_type
            FROM admins a
            INNER JOIN customers c ON a.user_id = c.id
            WHERE a.user_id = :user_id AND a.is_active = 1 AND c.is_active = 1
            LIMIT 1
        ");
        $stmt->execute([':user_id' => $userId]);
        if ($stmt->fetch()) {
            return 'admin';
        }
        
        // Organizer check
        $stmt = $pdo->prepare("
            SELECT 'organizer' as user_type
            FROM organizers o
            INNER JOIN customers c ON o.user_id = c.id
            WHERE o.user_id = :user_id AND o.is_active = 1 AND c.is_active = 1
            LIMIT 1
        ");
        $stmt->execute([':user_id' => $userId]);
        if ($stmt->fetch()) {
            return 'organizer';
        }
        
        // Venue check
        $stmt = $pdo->prepare("
            SELECT 'venue' as user_type
            FROM venues v
            INNER JOIN customers c ON v.user_id = c.id
            WHERE v.user_id = :user_id AND v.is_active = 1 AND c.is_active = 1
            LIMIT 1
        ");
        $stmt->execute([':user_id' => $userId]);
        if ($stmt->fetch()) {
            return 'venue';
        }
        
        // Default to customer
        return 'customer';
    } catch (Exception $e) {
        error_log('Error getting user type: ' . $e->getMessage());
        return 'customer';
    }
}

/**
 * Check if user has a specific permission
 */
function hasPermission($permissionSlug, $userId = null, $userType = null) {
    if (!isLoggedIn()) {
        return false;
    }
    
    $userId = $userId ?? $_SESSION['user_id'];
    $userType = $userType ?? getCurrentUserType($userId);
    
    try {
        $pdo = getDB();
        
        // Check if user is super_admin or admin (they have all permissions)
        // This works for customers table - checks if user is in admins table
        if ($userType === 'admin') {
            $stmt = $pdo->prepare("
                SELECT a.role 
                FROM admins a
                INNER JOIN customers c ON a.user_id = c.id
                WHERE a.user_id = :user_id AND a.is_active = 1 AND c.is_active = 1
            ");
            $stmt->execute([':user_id' => $userId]);
            $admin = $stmt->fetch();
            
            if ($admin && in_array($admin['role'], ['super_admin', 'admin'])) {
                // Super admin and admin have all permissions
                return true;
            }
        }

        // Backward compatible: some DBs may not have user_roles.user_type yet
        static $hasUserTypeColumn = null;
        if ($hasUserTypeColumn === null) {
            try {
                $check = $pdo->query("SHOW COLUMNS FROM user_roles LIKE 'user_type'");
                $hasUserTypeColumn = ($check && $check->rowCount() > 0);
            } catch (Exception $e) {
                $hasUserTypeColumn = false;
            }
        }

        // Check if user has the permission through their roles
        // Note: Even manager roles need explicit permissions assigned - no automatic grants
        $sql = "
            SELECT COUNT(*) as has_permission
            FROM user_roles ur
            INNER JOIN role_permissions rp ON ur.role_id = rp.role_id
            INNER JOIN permissions p ON rp.permission_id = p.id
            INNER JOIN roles r ON ur.role_id = r.id
            WHERE ur.user_id = :user_id
            AND p.slug = :permission_slug
            AND p.is_active = 1
            AND r.is_active = 1
        ";
        $params = [
            ':user_id' => $userId,
            ':permission_slug' => $permissionSlug
        ];
        if ($hasUserTypeColumn) {
            $sql .= " AND ur.user_type = :user_type";
            $params[':user_type'] = $userType;
        }

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch();
        
        return ($result && $result['has_permission'] > 0);
    } catch (Exception $e) {
        error_log('Error checking permission: ' . $e->getMessage());
        return false;
    }
}

/**
 * Check if user has any of the given permissions
 */
function hasAnyPermission($permissionSlugs, $userId = null) {
    foreach ($permissionSlugs as $slug) {
        if (hasPermission($slug, $userId)) {
            return true;
        }
    }
    return false;
}

/**
 * Check if user has all of the given permissions
 */
function hasAllPermissions($permissionSlugs, $userId = null) {
    foreach ($permissionSlugs as $slug) {
        if (!hasPermission($slug, $userId)) {
            return false;
        }
    }
    return true;
}

/**
 * Require a specific permission
 */
function requirePermission($permissionSlug, $errorMessage = 'Insufficient permissions') {
    if (!hasPermission($permissionSlug)) {
        $isApiRequest = (
            strpos($_SERVER['REQUEST_URI'] ?? '', '/api/') !== false ||
            (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)
        );
        
        if ($isApiRequest) {
            http_response_code(403);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => $errorMessage,
                'error' => 'Permission denied: ' . $permissionSlug
            ]);
            exit;
        }
        
        // Determine redirect URL based on user type
        $userType = getCurrentUserType();
        $baseUrl = getBaseUrlForRedirect();
        
        if ($userType === 'organizer') {
            header('Location: ' . $baseUrl . '/organizer/dashboard.php?error=' . urlencode($errorMessage));
        } elseif ($userType === 'venue') {
            header('Location: ' . $baseUrl . '/venue/dashboard.php?error=' . urlencode($errorMessage));
        } else {
            header('Location: ' . $baseUrl . '/admin/dashboard.php?error=' . urlencode($errorMessage));
        }
        exit;
    }
}

/**
 * Get all permissions for a user
 */
function getUserPermissions($userId = null, $userType = null) {
    if (!isLoggedIn()) {
        return [];
    }
    
    $userId = $userId ?? $_SESSION['user_id'];
    $userType = $userType ?? getCurrentUserType($userId);
    
    try {
        $pdo = getDB();
        
        // Check if user is super_admin or admin (only for admin type)
        if ($userType === 'admin') {
            $stmt = $pdo->prepare("
                SELECT a.role 
                FROM admins a
                INNER JOIN customers c ON a.user_id = c.id
                WHERE a.user_id = :user_id AND a.is_active = 1 AND c.is_active = 1
            ");
            $stmt->execute([':user_id' => $userId]);
            $admin = $stmt->fetch();
            
            if ($admin && in_array($admin['role'], ['super_admin', 'admin'])) {
                // Return all permissions for super_admin and admin
                $stmt = $pdo->query("SELECT slug FROM permissions WHERE is_active = 1");
                $permissions = $stmt->fetchAll(PDO::FETCH_COLUMN);
                return $permissions;
            }
        }
        
        // Backward compatible: some DBs may not have user_roles.user_type yet
        static $hasUserTypeColumn = null;
        if ($hasUserTypeColumn === null) {
            try {
                $check = $pdo->query("SHOW COLUMNS FROM user_roles LIKE 'user_type'");
                $hasUserTypeColumn = ($check && $check->rowCount() > 0);
            } catch (Exception $e) {
                $hasUserTypeColumn = false;
            }
        }

        // Get permissions from user's roles
        $sql = "
            SELECT DISTINCT p.slug
            FROM user_roles ur
            INNER JOIN role_permissions rp ON ur.role_id = rp.role_id
            INNER JOIN permissions p ON rp.permission_id = p.id
            INNER JOIN roles r ON ur.role_id = r.id
            WHERE ur.user_id = :user_id
            AND p.is_active = 1
            AND r.is_active = 1
        ";
        $params = [':user_id' => $userId];
        if ($hasUserTypeColumn) {
            $sql .= " AND ur.user_type = :user_type";
            $params[':user_type'] = $userType;
        }

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (Exception $e) {
        error_log('Error getting user permissions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get all roles for a user (works with customers table)
 */
function getUserRoles($userId = null, $userType = null) {
    if (!isLoggedIn()) {
        return [];
    }
    
    $userId = $userId ?? $_SESSION['user_id'];
    $userType = $userType ?? getCurrentUserType($userId);
    
    try {
        $pdo = getDB();
        
        $roles = [];
        
        // For admin users, check admin table first (from customers table)
        if ($userType === 'admin') {
            $stmt = $pdo->prepare("
                SELECT a.role 
                FROM admins a
                INNER JOIN customers c ON a.user_id = c.id
                WHERE a.user_id = :user_id AND a.is_active = 1 AND c.is_active = 1
            ");
            $stmt->execute([':user_id' => $userId]);
            $admin = $stmt->fetch();
            
            if ($admin) {
                $roles[] = $admin['role'];
            }
        }
        
        // Get roles from user_roles table
        // Check if user_type column exists first
        try {
            $checkColumn = $pdo->query("SHOW COLUMNS FROM user_roles LIKE 'user_type'");
            $hasUserType = $checkColumn->rowCount() > 0;
        } catch (Exception $e) {
            $hasUserType = false;
        }
        
        if ($hasUserType) {
            // Filter by user_type if column exists
            $stmt = $pdo->prepare("
                SELECT r.slug, r.name
                FROM user_roles ur
                INNER JOIN roles r ON ur.role_id = r.id
                WHERE ur.user_id = :user_id 
                AND ur.user_type = :user_type
                AND r.is_active = 1
            ");
            $stmt->execute([
                ':user_id' => $userId,
                ':user_type' => $userType
            ]);
        } else {
            // If user_type column doesn't exist, get all roles for user
            $stmt = $pdo->prepare("
                SELECT r.slug, r.name
                FROM user_roles ur
                INNER JOIN roles r ON ur.role_id = r.id
                WHERE ur.user_id = :user_id 
                AND r.is_active = 1
            ");
            $stmt->execute([
                ':user_id' => $userId
            ]);
        }
        $additionalRoles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($additionalRoles as $role) {
            if (!in_array($role['slug'], $roles)) {
                $roles[] = $role['slug'];
            }
        }
        
        return $roles;
    } catch (Exception $e) {
        error_log('Error getting user roles: ' . $e->getMessage());
        return [];
    }
}

/**
 * Check if user is super admin
 */
function isSuperAdmin($userId = null) {
    if (!isLoggedIn()) {
        return false;
    }
    
    $userId = $userId ?? $_SESSION['user_id'];
    
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            SELECT a.role 
            FROM admins a
            INNER JOIN customers c ON a.user_id = c.id
            WHERE a.user_id = :user_id AND a.role = 'super_admin' AND a.is_active = 1 AND c.is_active = 1
        ");
        $stmt->execute([':user_id' => $userId]);
        return $stmt->fetch() !== false;
    } catch (Exception $e) {
        error_log('Error checking super admin: ' . $e->getMessage());
        return false;
    }
}

/**
 * Get all roles
 */
function getAllRoles($includeInactive = false) {
    try {
        $pdo = getDB();
        $sql = "SELECT * FROM roles";
        if (!$includeInactive) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY is_system_role DESC, name ASC";
        $stmt = $pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log('Error getting all roles: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get all permissions
 */
function getAllPermissions($module = null, $includeInactive = false) {
    try {
        $pdo = getDB();
        $sql = "SELECT * FROM permissions";
        $conditions = [];
        if (!$includeInactive) {
            $conditions[] = "is_active = 1";
        }
        if ($module) {
            $conditions[] = "module = :module";
        }
        if (!empty($conditions)) {
            $sql .= " WHERE " . implode(" AND ", $conditions);
        }
        $sql .= " ORDER BY module, name ASC";
        
        $stmt = $pdo->prepare($sql);
        if ($module) {
            $stmt->execute([':module' => $module]);
        } else {
            $stmt->execute();
        }
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log('Error getting all permissions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get permissions grouped by module
 */
function getPermissionsByModule($includeInactive = false) {
    $permissions = getAllPermissions(null, $includeInactive);
    $grouped = [];
    
    foreach ($permissions as $permission) {
        $module = $permission['module'];
        if (!isset($grouped[$module])) {
            $grouped[$module] = [];
        }
        $grouped[$module][] = $permission;
    }
    
    return $grouped;
}

/**
 * Get permissions for a role
 */
function getRolePermissions($roleId) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            SELECT p.*
            FROM role_permissions rp
            INNER JOIN permissions p ON rp.permission_id = p.id
            WHERE rp.role_id = :role_id AND p.is_active = 1
            ORDER BY p.module, p.name
        ");
        $stmt->execute([':role_id' => $roleId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log('Error getting role permissions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get role by ID
 */
function getRoleById($roleId) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("SELECT * FROM roles WHERE id = :id");
        $stmt->execute([':id' => $roleId]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log('Error getting role: ' . $e->getMessage());
        return null;
    }
}

/**
 * Get role by slug
 */
function getRoleBySlug($slug) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("SELECT * FROM roles WHERE slug = :slug");
        $stmt->execute([':slug' => $slug]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log('Error getting role by slug: ' . $e->getMessage());
        return null;
    }
}

