<?php
/**
 * Authentication helper functions
 */

require_once __DIR__ . '/../config.php';

/**
 * Get database connection
 */
function getDB() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]
            );
        } catch (PDOException $e) {
            error_log('Database connection failed: ' . $e->getMessage());
            throw new Exception('Database connection failed');
        }
    }
    
    return $pdo;
}

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Get current user data
 */
function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = :id AND is_active = 1");
        $stmt->execute([':id' => $_SESSION['user_id']]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log('Error getting current user: ' . $e->getMessage());
        return null;
    }
}

/**
 * Login user (set session)
 */
function loginUser($userId, $userData = null) {
    $_SESSION['user_id'] = $userId;
    $_SESSION['user_email'] = $userData['email'] ?? null;
    $_SESSION['user_name'] = $userData['full_name'] ?? ($userData['first_name'] . ' ' . $userData['last_name'] ?? '');
    $_SESSION['user_provider'] = $userData['provider'] ?? null;
    
    // Update last login
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("UPDATE customers SET last_login = NOW() WHERE id = :id");
        $stmt->execute([':id' => $userId]);
    } catch (Exception $e) {
        error_log('Error updating last login: ' . $e->getMessage());
    }
}

/**
 * Logout user
 */
function logoutUser() {
    $_SESSION = array();
    
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    
    session_destroy();
}

/**
 * Create or update user from OAuth data
 */
function createOrUpdateUser($provider, $providerData) {
    try {
        $pdo = getDB();
        
        // Extract user data based on provider
        $email = $providerData['email'] ?? null;
        $providerId = $providerData['id'] ?? $providerData['sub'] ?? null;
        $firstName = $providerData['given_name'] ?? $providerData['first_name'] ?? '';
        $lastName = $providerData['family_name'] ?? $providerData['last_name'] ?? '';
        $fullName = $providerData['name'] ?? ($firstName . ' ' . $lastName);
        $profilePicture = $providerData['picture'] ?? $providerData['profile_picture'] ?? null;
        $accessToken = $providerData['access_token'] ?? null;
        $refreshToken = $providerData['refresh_token'] ?? null;
        
        if (!$email || !$providerId) {
            throw new Exception('Missing required user data');
        }
        
        // Check if user exists
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE email = :email OR (provider = :provider AND provider_id = :provider_id)");
        $stmt->execute([
            ':email' => $email,
            ':provider' => $provider,
            ':provider_id' => $providerId
        ]);
        $existingUser = $stmt->fetch();
        
        if ($existingUser) {
            // Update existing user
            $stmt = $pdo->prepare("
                UPDATE customers 
                SET first_name = :first_name,
                    last_name = :last_name,
                    full_name = :full_name,
                    profile_picture = :profile_picture,
                    provider = :provider,
                    provider_id = :provider_id,
                    access_token = :access_token,
                    refresh_token = :refresh_token,
                    last_login = NOW(),
                    updated_at = NOW()
                WHERE id = :id
            ");
            
            $stmt->execute([
                ':id' => $existingUser['id'],
                ':first_name' => $firstName,
                ':last_name' => $lastName,
                ':full_name' => trim($fullName),
                ':profile_picture' => $profilePicture,
                ':provider' => $provider,
                ':provider_id' => $providerId,
                ':access_token' => $accessToken,
                ':refresh_token' => $refreshToken
            ]);
            
            return $existingUser['id'];
        } else {
            // Create new user
            $stmt = $pdo->prepare("
                INSERT INTO customers (
                    email, first_name, last_name, full_name, profile_picture,
                    provider, provider_id, access_token, refresh_token, created_at
                ) VALUES (
                    :email, :first_name, :last_name, :full_name, :profile_picture,
                    :provider, :provider_id, :access_token, :refresh_token, NOW()
                )
            ");
            
            $stmt->execute([
                ':email' => $email,
                ':first_name' => $firstName,
                ':last_name' => $lastName,
                ':full_name' => trim($fullName),
                ':profile_picture' => $profilePicture,
                ':provider' => $provider,
                ':provider_id' => $providerId,
                ':access_token' => $accessToken,
                ':refresh_token' => $refreshToken
            ]);
            
            return $pdo->lastInsertId();
        }
    } catch (Exception $e) {
        error_log('Error creating/updating user: ' . $e->getMessage());
        throw $e;
    }
}
