// Get event ID from URL
const urlParams = new URLSearchParams(window.location.search);
const eventId = urlParams.get('id');

let eventData = null;

// Load event details on page load
document.addEventListener('DOMContentLoaded', function() {
    if (!eventId) {
        document.body.innerHTML = '<div class="container"><h1>Event not found</h1><p><a href="events.html">Back to Events</a></p></div>';
        return;
    }
    
    loadEventDetails(eventId);
});

// Load event details from API
function loadEventDetails(id) {
    fetch(`api/public/eventdetailapi.php?id=${encodeURIComponent(id)}`)
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data && data.data.event) {
                eventData = data.data.event;
                populateEventDetails(data.data.event);
            } else {
                document.body.innerHTML = '<div class="container"><h1>Event not found</h1><p><a href="events.html">Back to Events</a></p></div>';
            }
        })
        .catch(error => {
            console.error('Error loading event:', error);
            document.body.innerHTML = '<div class="container"><h1>Error loading event</h1><p><a href="events.html">Back to Events</a></p></div>';
        });
}

// Populate event details on the page
function populateEventDetails(event) {
    // Event type badge
    const eventType = capitalizeFirst(event.event_type || 'Event');
    document.getElementById('detailEventType').textContent = eventType.toUpperCase();
    
    // Ticket badge
    if (event.ticket_available) {
        document.getElementById('detailTicketBadge').style.display = 'inline-flex';
    }
    
    // Banner Image
    if (event.banner_image) {
        const bannerWrapper = document.getElementById('eventBannerWrapper');
        const bannerImage = document.getElementById('eventBannerImage');
        if (bannerWrapper && bannerImage) {
            bannerImage.src = event.banner_image;
            bannerImage.alt = event.name || 'Event Banner';
            bannerImage.onerror = function() {
                this.style.display = 'none';
                bannerWrapper.style.display = 'none';
            };
            bannerWrapper.style.display = 'block';
        }
    }
    
    // Title
    document.getElementById('eventDetailTitle').textContent = event.name || 'Untitled Event';
    
    // Description
    document.getElementById('eventDetailDescription').textContent = event.description || 'No description available.';
    
    // Date and location
    const startDate = event.start_date ? new Date(event.start_date) : null;
    const endDate = event.end_date ? new Date(event.end_date) : null;
    const dateRange = formatDetailDateRange(startDate, endDate);
    document.getElementById('eventDetailDate').textContent = dateRange;
    
    const city = event.city || '';
    const country = event.country || '';
    const venue = event.venue_name || (event.venue && event.venue !== 'TBA' ? event.venue : '');
    const venueAddress = event.venue_address || '';
    const fullVenue = venue ? (venueAddress ? `${venue}, ${venueAddress}` : venue) : '';
    const location = fullVenue ? `${fullVenue}${city ? ', ' + city : ''}${country ? ', ' + country : ''}` : (city ? `${city}, ${country}` : country || 'TBA');
    document.getElementById('eventDetailLocation').textContent = location;
    
    // Populate Event Information Section
    populateEventInfo(event, startDate, endDate);
    
    // Populate About the Event
    const aboutContent = event.description || 'No description available.';
    document.getElementById('eventAboutContent').innerHTML = `<p>${escapeHtml(aboutContent).replace(/\n/g, '</p><p>')}</p>`;
    
    // Populate Terms and Conditions
    const termsContent = event.terms_conditions || event.refund_policy || 'No terms and conditions specified.';
    document.getElementById('eventTermsContent').innerHTML = `<p>${escapeHtml(termsContent).replace(/\n/g, '</p><p>')}</p>`;
    
    // Statistics cards
    populateStatsCards(event, startDate, endDate);
    
    // Who Attends
    document.getElementById('whoAttends').textContent = event.attendee_profiles || 'General public and event enthusiasts';
    
    // Sponsors
    populateSponsors(event.sponsors || '');
    
    // Sidebar
    document.getElementById('sidebarDate').textContent = formatSidebarDate(startDate);
    const price = event.pricing || (event.ticket_available ? 'Free' : 'TBA');
    document.getElementById('sidebarPrice').textContent = price;
    
    const locationLabel = venue || city || 'Location';
    const locationValue = city && country ? `${city}, ${country}` : country || 'TBA';
    document.getElementById('sidebarLocationLabel').textContent = locationLabel;
    document.getElementById('sidebarLocationValue').textContent = locationValue;
}

// Populate statistics cards
function populateStatsCards(event, startDate, endDate) {
    const attendees = event.attendees_range || '1000+';
    const country = event.country || 'TBA';
    
    // Calculate duration
    let duration = 'TBA';
    if (startDate && endDate) {
        const diffTime = Math.abs(endDate - startDate);
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)) + 1;
        duration = diffDays === 1 ? '1 Day' : `${diffDays} Days`;
    } else if (startDate) {
        duration = '1 Day';
    }
    
    const price = event.pricing || (event.ticket_available ? 'Free' : 'TBA');
    
    const stats = [
        { icon: '👥', value: attendees, label: 'Expected Attendees' },
        { icon: '🏢', value: country, label: 'Location' },
        { icon: '⏰', value: duration, label: 'Duration' },
        { icon: '💰', value: price, label: 'Starting Price' }
    ];
    
    const statsGrid = document.getElementById('eventStatsGrid');
    statsGrid.innerHTML = stats.map(stat => `
        <div class="stat-card">
            <div class="stat-icon">${stat.icon}</div>
            <div class="stat-value">${escapeHtml(stat.value)}</div>
            <div class="stat-label">${escapeHtml(stat.label)}</div>
        </div>
    `).join('');
}

// Populate sponsors
function populateSponsors(sponsorsStr) {
    const sponsorsList = document.getElementById('sponsorsList');
    
    if (!sponsorsStr || sponsorsStr.trim() === '') {
        sponsorsList.innerHTML = '<p class="no-sponsors">No sponsors listed</p>';
        return;
    }
    
    const sponsors = sponsorsStr.split(',').map(s => s.trim()).filter(s => s);
    
    if (sponsors.length === 0) {
        sponsorsList.innerHTML = '<p class="no-sponsors">No sponsors listed</p>';
        return;
    }
    
    sponsorsList.innerHTML = sponsors.map(sponsor => `
        <span class="sponsor-tag">${escapeHtml(sponsor)}</span>
    `).join('');
}

// Format date range for detail page
function formatDetailDateRange(startDate, endDate) {
    if (!startDate) return 'Date TBA';
    
    const formatDate = (date) => {
        return date.toLocaleDateString('en-US', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    };
    
    const start = formatDate(startDate);
    if (endDate && endDate.getTime() !== startDate.getTime()) {
        const end = formatDate(endDate);
        return `${start} - ${end}`;
    }
    return start;
}

// Format sidebar date
function formatSidebarDate(date) {
    if (!date) return 'TBA';
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Populate Event Information Section
function populateEventInfo(event, startDate, endDate) {
    // Date: Thursday, 15 January 2026
    if (startDate) {
        const dateStr = startDate.toLocaleDateString('en-US', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
        document.getElementById('eventInfoDate').textContent = dateStr;
    } else {
        document.getElementById('eventInfoDate').textContent = 'TBA';
    }
    
    // Time: 9:30 PM
    if (startDate) {
        const timeStr = startDate.toLocaleTimeString('en-US', {
            hour: 'numeric',
            minute: '2-digit',
            hour12: true
        });
        document.getElementById('eventInfoTime').textContent = timeStr;
    } else {
        document.getElementById('eventInfoTime').textContent = 'TBA';
    }
    
    // Duration: 2 Hours
    let duration = 'TBA';
    if (startDate && endDate) {
        const diffTime = Math.abs(endDate - startDate);
        const diffHours = Math.ceil(diffTime / (1000 * 60 * 60));
        const diffDays = Math.floor(diffHours / 24);
        if (diffDays > 0) {
            duration = diffDays === 1 ? '1 Day' : `${diffDays} Days`;
        } else if (diffHours > 0) {
            duration = diffHours === 1 ? '1 Hour' : `${diffHours} Hours`;
        } else {
            duration = 'Less than 1 Hour';
        }
    } else if (startDate) {
        duration = '1 Hour';
    }
    document.getElementById('eventInfoDuration').textContent = duration;
    
    // Age Limit: 18 years and above
    const ageLimit = event.age_restriction || 'No age restriction';
    document.getElementById('eventInfoAgeLimit').textContent = ageLimit;
    
    // Language: English, Hindi
    const language = event.language || 'Not specified';
    document.getElementById('eventInfoLanguage').textContent = language;
    
    // Genre: Comedy
    const genre = event.genre || event.category || event.event_type || 'Not specified';
    document.getElementById('eventInfoGenre').textContent = genre;
    
    // Venue: The Laugh Store, DLF Cyberhub, Gurugram
    const venueName = event.venue_name || '';
    const venueAddress = event.venue_address || '';
    const venueCity = event.city || '';
    const venueCountry = event.country || '';
    let venueText = '';
    if (venueName) {
        venueText = venueName;
        if (venueAddress) {
            venueText += ', ' + venueAddress;
        }
        if (venueCity) {
            venueText += ', ' + venueCity;
        }
        if (venueCountry && venueCountry !== venueCity) {
            venueText += ', ' + venueCountry;
        }
    } else if (venueCity) {
        venueText = venueCity + (venueCountry ? ', ' + venueCountry : '');
    } else {
        venueText = 'TBA';
    }
    document.getElementById('eventInfoVenue').textContent = venueText;
}

// Handle Get Tickets button
function handleGetTickets() {
    if (eventData && eventData.website_url) {
        window.open(eventData.website_url, '_blank', 'noopener,noreferrer');
    } else {
        alert('Ticket information not available. Please check the event website.');
    }
}

// Handle Share Event button
function handleShareEvent() {
    if (navigator.share) {
        navigator.share({
            title: eventData.name,
            text: eventData.description,
            url: window.location.href
        }).catch(err => console.log('Error sharing', err));
    } else {
        // Fallback: copy to clipboard
        navigator.clipboard.writeText(window.location.href).then(() => {
            alert('Event link copied to clipboard!');
        }).catch(() => {
            // Fallback: show URL
            prompt('Copy this link:', window.location.href);
        });
    }
}

// Capitalize first letter
function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}

// Escape HTML
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
