<?php
/**
 * Events API - Public endpoint
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';

try {
    $pdo = getDB();
    
    // Get query parameters
    $eventType = $_GET['type'] ?? null;
    $country = $_GET['country'] ?? null;
    $category = $_GET['category'] ?? null;
    $city = $_GET['city'] ?? null;
    $search = $_GET['search'] ?? null;
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = max(1, min(100, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = ["e.status = 'approved'", "e.status = 'live'"];
    $params = [];
    
    if ($eventType && $eventType !== 'all') {
        $where[] = "e.event_type = :event_type";
        $params[':event_type'] = $eventType;
    }
    
    if ($country) {
        $where[] = "e.country = :country";
        $params[':country'] = $country;
    }
    
    if ($category) {
        $where[] = "e.category = :category";
        $params[':category'] = $category;
    }
    
    if ($city) {
        $where[] = "e.city = :city";
        $params[':city'] = $city;
    }
    
    if ($search) {
        $where[] = "(e.name LIKE :search OR e.description LIKE :search)";
        $params[':search'] = '%' . $search . '%';
    }
    
    $whereClause = '(' . implode(' OR ', array_slice($where, 0, 2)) . ')';
    if (count($where) > 2) {
        $whereClause .= ' AND ' . implode(' AND ', array_slice($where, 2));
    }
    
    // Get total count
    $countStmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM events e
        WHERE {$whereClause}
    ");
    $countStmt->execute($params);
    $total = $countStmt->fetch()['total'];
    
    // Get events
    $stmt = $pdo->prepare("
        SELECT 
            e.*,
            o.company_name as organizer_name,
            v.venue_name
        FROM events e
        LEFT JOIN organizers o ON e.organizer_id = o.id
        LEFT JOIN venues v ON e.venue_id = v.id
        WHERE {$whereClause}
        ORDER BY e.start_date ASC, e.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    
    $events = $stmt->fetchAll();
    
    // Format events
    foreach ($events as &$event) {
        $event['image_gallery'] = $event['image_gallery'] ? json_decode($event['image_gallery'], true) : [];
        $event['artist_lineup'] = $event['artist_lineup'] ? json_decode($event['artist_lineup'], true) : [];
    }
    
    sendSuccess([
        'events' => $events,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Events API Error: ' . $e->getMessage());
    sendError('Failed to load events', 500);
}

