<?php
/**
 * Event Detail API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';

try {
    $pdo = getDB();
    
    $eventId = $_GET['id'] ?? null;
    
    if (!$eventId) {
        sendError('Event ID is required', 400);
    }
    
    // Get event details
    $stmt = $pdo->prepare("
        SELECT 
            e.*,
            o.company_name as organizer_name,
            o.description as organizer_description,
            o.logo as organizer_logo,
            v.venue_name,
            v.address as venue_address,
            v.capacity as venue_capacity,
            v.seating_zones as venue_seating_zones,
            v.floor_plan as venue_floor_plan,
            COALESCE(e.language, 'Not specified') as language,
            COALESCE(e.genre, e.category, e.event_type, 'Not specified') as genre,
            COALESCE(e.terms_conditions, e.refund_policy, 'No terms and conditions specified.') as terms_conditions
        FROM events e
        LEFT JOIN organizers o ON e.organizer_id = o.id
        LEFT JOIN venues v ON e.venue_id = v.id
        WHERE e.id = :id AND (e.status = 'approved' OR e.status = 'live')
    ");
    $stmt->execute([':id' => $eventId]);
    $event = $stmt->fetch();
    
    if (!$event) {
        sendError('Event not found', 404);
    }
    
    // Get ticket types
    $ticketStmt = $pdo->prepare("
        SELECT * FROM ticket_types 
        WHERE event_id = :event_id AND is_active = 1
        ORDER BY price ASC
    ");
    $ticketStmt->execute([':event_id' => $eventId]);
    $tickets = $ticketStmt->fetchAll();
    
    // Format event data
    $event['image_gallery'] = $event['image_gallery'] ? json_decode($event['image_gallery'], true) : [];
    $event['artist_lineup'] = $event['artist_lineup'] ? json_decode($event['artist_lineup'], true) : [];
    $event['event_schedule'] = $event['event_schedule'] ? json_decode($event['event_schedule'], true) : [];
    $event['seating_layout'] = $event['seating_layout'] ? json_decode($event['seating_layout'], true) : [];
    $event['venue_seating_zones'] = $event['venue_seating_zones'] ? json_decode($event['venue_seating_zones'], true) : [];
    
    // Get reviews
    $reviewStmt = $pdo->prepare("
        SELECT r.*, c.full_name as customer_name, c.profile_picture
        FROM reviews r
        INNER JOIN customers c ON r.customer_id = c.id
        WHERE r.event_id = :event_id AND r.is_approved = 1
        ORDER BY r.created_at DESC
        LIMIT 10
    ");
    $reviewStmt->execute([':event_id' => $eventId]);
    $reviews = $reviewStmt->fetchAll();
    
    // Calculate average rating
    $avgRatingStmt = $pdo->prepare("
        SELECT AVG(rating) as avg_rating, COUNT(*) as total_reviews
        FROM reviews
        WHERE event_id = :event_id AND is_approved = 1
    ");
    $avgRatingStmt->execute([':event_id' => $eventId]);
    $ratingData = $avgRatingStmt->fetch();
    
    sendSuccess([
        'event' => $event,
        'tickets' => $tickets,
        'reviews' => $reviews,
        'rating' => [
            'average' => round($ratingData['avg_rating'] ?? 0, 1),
            'total' => $ratingData['total_reviews'] ?? 0
        ]
    ]);
} catch (Exception $e) {
    error_log('Event Detail API Error: ' . $e->getMessage());
    sendError('Failed to load event details', 500);
}

