<?php
/**
 * Organizer Sponsors API
 * Handles CRUD operations for sponsors (organizer-owned only)
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Check if sponsors table exists
    try {
        $pdo->query("SELECT 1 FROM sponsors LIMIT 1");
    } catch (Exception $e) {
        sendError('Sponsors module not initialized. Please run the migration first.', 500);
    }
    
    // Soft-delete backward compatibility
    try {
        $check = $pdo->query("SHOW COLUMNS FROM sponsors LIKE 'deleted_at'");
        $hasDeletedAt = ($check && $check->rowCount() > 0);
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    $sponsorId = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if ($method === 'GET') {
        // List sponsors - only organizer's own sponsors
        if (!hasPermission('organizer.sponsors.view')) {
            sendError('You do not have permission to view sponsors.', 403);
        }
        
        $statusFilter = $_GET['status'] ?? 'all';
        $search = $_GET['search'] ?? '';
        
        $sql = "SELECT * FROM sponsors WHERE organizer_id = :organizer_id";
        $params = [':organizer_id' => $organizer['id']];
        
        // Exclude soft-deleted
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        // Status filter
        if ($statusFilter !== 'all') {
            $sql .= " AND status = :status";
            $params[':status'] = $statusFilter;
        }
        
        // Search filter
        if (!empty($search)) {
            $sql .= " AND name LIKE :search";
            $params[':search'] = '%' . $search . '%';
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $sponsors = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($sponsors);
        
    } elseif ($method === 'POST' && $action !== 'update' && $action !== 'delete') {
        // Create sponsor
        if (!hasPermission('organizer.sponsors.create')) {
            sendError('You do not have permission to create sponsors.', 403);
        }
        
        $data = getRequestData();
        $errors = validateRequired($data, ['name']);
        if (!empty($errors)) {
            sendError('Validation failed', 400, $errors);
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO sponsors (
                organizer_id, name, description, logo_image, website_url, status, is_active, created_at
            ) VALUES (
                :organizer_id, :name, :description, :logo_image, :website_url, 'unpublished', 1, NOW()
            )
        ");
        
        $stmt->execute([
            ':organizer_id' => $organizer['id'],
            ':name' => $data['name'],
            ':description' => $data['description'] ?? null,
            ':logo_image' => $data['logo_image'] ?? null,
            ':website_url' => $data['website_url'] ?? null
        ]);
        
        $sponsorId = $pdo->lastInsertId();
        
        sendSuccess([
            'message' => 'Sponsor created successfully',
            'sponsor_id' => $sponsorId
        ]);
        
    } elseif ($method === 'GET' && $sponsorId) {
        // Get single sponsor
        if (!hasPermission('organizer.sponsors.view')) {
            sendError('You do not have permission to view sponsors.', 403);
        }
        
        $sql = "SELECT * FROM sponsors WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
        $sponsor = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$sponsor) {
            sendError('Sponsor not found or you do not have permission to view this sponsor.', 404);
        }
        
        sendSuccess($sponsor);
        
    } elseif (($method === 'PUT' || $method === 'PATCH' || ($method === 'POST' && $action === 'update')) && $sponsorId) {
        // Update sponsor
        if (!hasPermission('organizer.sponsors.edit')) {
            sendError('You do not have permission to edit sponsors.', 403);
        }
        
        // Verify ownership
        $checkSql = "SELECT id FROM sponsors WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Sponsor not found or you do not have permission to edit this sponsor.', 404);
        }
        
        $data = getRequestData();
        
        // Build update query dynamically
        $updateFields = [];
        $updateParams = [
            ':id' => $sponsorId,
            ':organizer_id' => $organizer['id']
        ];
        
        $allowedFields = ['name', 'description', 'logo_image', 'website_url', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field]) && $data[$field] !== '') {
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = $data[$field];
            } elseif (isset($data[$field]) && $data[$field] === '' && in_array($field, ['description', 'logo_image', 'website_url'])) {
                // Allow blank for optional fields (set to NULL)
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = null;
            }
        }
        
        if (empty($updateFields)) {
            sendError('No fields to update', 400);
        }
        
        $updateFields[] = "updated_at = NOW()";
        $sql = "UPDATE sponsors SET " . implode(', ', $updateFields) . " WHERE id = :id AND organizer_id = :organizer_id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($updateParams);
        
        sendSuccess(['message' => 'Sponsor updated successfully']);
        
    } elseif (($method === 'DELETE' || ($method === 'POST' && $action === 'delete')) && $sponsorId) {
        // Soft delete sponsor
        if (!hasPermission('organizer.sponsors.delete')) {
            sendError('You do not have permission to delete sponsors.', 403);
        }
        
        // Verify ownership
        $checkSql = "SELECT id FROM sponsors WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Sponsor not found or you do not have permission to delete this sponsor.', 404);
        }
        
        if ($hasDeletedAt) {
            // Soft delete
            $stmt = $pdo->prepare("UPDATE sponsors SET deleted_at = NOW(), status = 'unpublished' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
        } else {
            // Fallback: just set is_active = 0
            $stmt = $pdo->prepare("UPDATE sponsors SET is_active = 0, status = 'unpublished' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $sponsorId, ':organizer_id' => $organizer['id']]);
        }
        
        sendSuccess(['message' => 'Sponsor deleted successfully']);
        
    } else {
        sendError('Invalid request', 400);
    }
    
} catch (Exception $e) {
    error_log('Sponsor API Error: ' . $e->getMessage());
    sendError('An error occurred: ' . $e->getMessage(), 500);
}


