<?php
/**
 * Organizer Sales & Finance API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

// Check permission to view sales
if (!hasPermission('organizer.sales.view')) {
    sendError('You do not have permission to view sales.', 403);
}

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Get sales statistics
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(DISTINCT b.id) as total_bookings,
            SUM(b.final_amount) as total_revenue,
            COUNT(DISTINCT t.id) as total_tickets_sold,
            AVG(b.final_amount) as avg_booking_value
        FROM bookings b
        INNER JOIN events e ON b.event_id = e.id
        LEFT JOIN tickets t ON b.id = t.booking_id
        WHERE e.organizer_id = :organizer_id AND b.payment_status = 'completed'
    ");
    $stmt->execute([':organizer_id' => $organizer['id']]);
    $sales_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get recent bookings
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 50);
    $offset = ($page - 1) * $limit;
    
    $stmt = $pdo->prepare("
        SELECT 
            b.*,
            e.name as event_name,
            c.full_name as customer_name,
            c.email as customer_email,
            COUNT(t.id) as ticket_count
        FROM bookings b
        INNER JOIN events e ON b.event_id = e.id
        INNER JOIN customers c ON b.customer_id = c.id
        LEFT JOIN tickets t ON b.id = t.booking_id
        WHERE e.organizer_id = :organizer_id
        GROUP BY b.id
        ORDER BY b.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(':organizer_id', $organizer['id']);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $bookings = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get total count
    $countStmt = $pdo->prepare("
        SELECT COUNT(DISTINCT b.id) as total 
        FROM bookings b
        INNER JOIN events e ON b.event_id = e.id
        WHERE e.organizer_id = :organizer_id
    ");
    $countStmt->execute([':organizer_id' => $organizer['id']]);
    $total = (int)$countStmt->fetch()['total'];
    
    sendSuccess([
        'sales_stats' => $sales_stats,
        'bookings' => $bookings,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Organizer Sales API Error: ' . $e->getMessage());
    sendError('Failed to load sales data', 500);
}

