<?php
/**
 * Organizer Refunds API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

// Check permission to view refunds
if (!hasPermission('organizer.refunds.view')) {
    sendError('You do not have permission to view refunds.', 403);
}

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Get refund requests
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 50);
    $offset = ($page - 1) * $limit;
    
    $stmt = $pdo->prepare("
        SELECT 
            r.*,
            b.booking_reference,
            e.name as event_name,
            c.full_name as customer_name,
            c.email as customer_email
        FROM refunds r
        INNER JOIN bookings b ON r.booking_id = b.id
        INNER JOIN events e ON b.event_id = e.id
        INNER JOIN customers c ON r.requested_by = c.id
        WHERE e.organizer_id = :organizer_id
        ORDER BY r.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(':organizer_id', $organizer['id']);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $refunds = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get total count
    $countStmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM refunds r
        INNER JOIN bookings b ON r.booking_id = b.id
        INNER JOIN events e ON b.event_id = e.id
        WHERE e.organizer_id = :organizer_id
    ");
    $countStmt->execute([':organizer_id' => $organizer['id']]);
    $total = (int)$countStmt->fetch()['total'];
    
    sendSuccess([
        'refunds' => $refunds,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Organizer Refunds API Error: ' . $e->getMessage());
    sendError('Failed to load refunds', 500);
}

