<?php
/**
 * Organizer Login API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';

$data = getRequestData();

$errors = validateRequired($data, ['email', 'password']);
if (!empty($errors)) {
    sendError('Validation failed', 400, $errors);
}

try {
    $pdo = getDB();
    
    // Find user by email
    $stmt = $pdo->prepare("SELECT * FROM customers WHERE email = :email AND is_active = 1");
    $stmt->execute([':email' => $data['email']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendError('Invalid email or password', 401);
    }
    
    // Check if user has organizer role
    $organizerStmt = $pdo->prepare("
        SELECT * FROM organizers 
        WHERE user_id = :user_id AND is_active = 1
    ");
    $organizerStmt->execute([':user_id' => $user['id']]);
    $organizer = $organizerStmt->fetch();
    
    if (!$organizer) {
        sendError('You do not have organizer access. Please register as an organizer first.', 403);
    }
    
    // Verify password
    if (!empty($user['password_hash'])) {
        if (!password_verify($data['password'], $user['password_hash'])) {
            sendError('Invalid email or password', 401);
        }
    } else {
        // If no password is set, require password setup
        sendError('Please set your password first', 401);
    }
    
    // Login user
    loginUser($user['id'], $user);
    
    // Determine correct redirect URL using BASE_URL constant
    $redirectUrl = BASE_URL . '/organizer/dashboard.php';
    
    sendSuccess([
        'user' => [
            'id' => $user['id'],
            'email' => $user['email'],
            'name' => $user['full_name']
        ],
        'redirect' => $redirectUrl
    ]);
} catch (Exception $e) {
    error_log('Organizer Login API Error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    sendError('Login failed: ' . $e->getMessage(), 500);
}

