<?php
/**
 * Event-Sponsor Mapping API
 * Handles linking/unlinking sponsors to events
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Check if event_sponsors table exists
    try {
        $pdo->query("SELECT 1 FROM event_sponsors LIMIT 1");
    } catch (Exception $e) {
        sendError('Event-Sponsor mapping module not initialized. Please run the migration first.', 500);
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    $eventId = isset($_GET['event_id']) ? (int)$_GET['event_id'] : null;
    
    if ($method === 'GET' && $eventId) {
        // Get sponsors for an event
        if (!hasPermission('organizer.events.view')) {
            sendError('You do not have permission to view events.', 403);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to view this event.', 404);
        }
        
        // Get sponsors linked to this event
        $stmt = $pdo->prepare("
            SELECT s.* FROM sponsors s
            INNER JOIN event_sponsors es ON s.id = es.sponsor_id
            WHERE es.event_id = :event_id
              AND s.organizer_id = :organizer_id
              AND s.deleted_at IS NULL
            ORDER BY s.name ASC
        ");
        $stmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        $sponsors = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($sponsors);
        
    } elseif ($method === 'POST' && $action === 'link') {
        // Link sponsor(s) to event
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events.', 403);
        }
        
        $data = getRequestData();
        $eventId = $data['event_id'] ?? null;
        $sponsorIds = $data['sponsor_ids'] ?? [];
        
        if (!$eventId) {
            sendError('Event ID is required', 400);
        }
        
        if (!is_array($sponsorIds) || empty($sponsorIds)) {
            sendError('At least one sponsor ID is required', 400);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Verify all sponsors belong to this organizer
        $placeholders = implode(',', array_fill(0, count($sponsorIds), '?'));
        $checkSponsorsStmt = $pdo->prepare("
            SELECT id FROM sponsors 
            WHERE id IN ($placeholders) 
              AND organizer_id = ? 
              AND deleted_at IS NULL
        ");
        $sponsorIdsInt = array_map('intval', $sponsorIds);
        $params = array_merge($sponsorIdsInt, [(int)$organizer['id']]);
        $checkSponsorsStmt->execute($params);
        $validSponsors = $checkSponsorsStmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (count($validSponsors) !== count($sponsorIds)) {
            sendError('One or more sponsors not found or do not belong to you.', 400);
        }
        
        // Link sponsors (ignore duplicates)
        $linked = 0;
        $stmt = $pdo->prepare("INSERT IGNORE INTO event_sponsors (event_id, sponsor_id) VALUES (:event_id, :sponsor_id)");
        
        foreach ($sponsorIds as $sponsorId) {
            $stmt->execute([':event_id' => $eventId, ':sponsor_id' => $sponsorId]);
            if ($stmt->rowCount() > 0) {
                $linked++;
            }
        }
        
        sendSuccess([
            'message' => "Successfully linked $linked sponsor(s) to event",
            'linked_count' => $linked
        ]);
        
    } elseif ($method === 'POST' && $action === 'unlink') {
        // Unlink sponsor from event
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events.', 403);
        }
        
        $data = getRequestData();
        $eventId = $data['event_id'] ?? null;
        $sponsorId = $data['sponsor_id'] ?? null;
        
        if (!$eventId || !$sponsorId) {
            sendError('Event ID and Sponsor ID are required', 400);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Remove link
        $stmt = $pdo->prepare("DELETE FROM event_sponsors WHERE event_id = :event_id AND sponsor_id = :sponsor_id");
        $stmt->execute([':event_id' => $eventId, ':sponsor_id' => $sponsorId]);
        
        sendSuccess(['message' => 'Sponsor unlinked from event successfully']);
        
    } elseif ($method === 'GET' && !$eventId) {
        // Get available sponsors for organizer (for dropdown)
        if (!hasPermission('organizer.sponsors.view')) {
            sendError('You do not have permission to view sponsors.', 403);
        }
        
        $statusFilter = $_GET['status'] ?? 'all';
        
        $sql = "SELECT id, name, logo_image, status FROM sponsors WHERE organizer_id = :organizer_id AND deleted_at IS NULL";
        $params = [':organizer_id' => $organizer['id']];
        
        if ($statusFilter === 'published') {
            $sql .= " AND status = 'published'";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $sponsors = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($sponsors);
        
    } else {
        sendError('Invalid request', 400);
    }
    
} catch (Exception $e) {
    error_log('Event-Sponsor API Error: ' . $e->getMessage());
    sendError('An error occurred: ' . $e->getMessage(), 500);
}


