<?php
/**
 * Organizer Events API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

// Check permission to view events
if (!hasPermission('organizer.events.view')) {
    sendError('You do not have permission to view events.', 403);
}

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();

    // Soft-delete backward compatibility: deleted_at may not exist if migration wasn't run
    try {
        $check = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
        $hasDeletedAt = ($check && $check->rowCount() > 0);
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    // Get filter parameters
    $status = $_GET['status'] ?? 'all';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 100);
    $offset = ($page - 1) * $limit;
    
    // Build query - exclude soft-deleted events if supported
    $where = ["e.organizer_id = :organizer_id"];
    if ($hasDeletedAt) {
        $where[] = "e.deleted_at IS NULL";
    }
    $params = [':organizer_id' => $organizer['id']];
    
    if ($status !== 'all') {
        $where[] = "e.status = :status";
        $params[':status'] = $status;
    }
    
    $whereClause = 'WHERE ' . implode(' AND ', $where);
    
    // Get total count
    $countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM events e {$whereClause}");
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch()['total'];
    
    // Get events
    $stmt = $pdo->prepare("
        SELECT e.*, v.venue_name
        FROM events e
        LEFT JOIN venues v ON e.venue_id = v.id
        {$whereClause}
        ORDER BY e.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccess([
        'events' => $events,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Organizer Events API Error: ' . $e->getMessage());
    sendError('Failed to load events', 500);
}

