<?php
/**
 * Event-Artist Mapping API
 * Handles linking/unlinking artists to events
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Check if event_artists table exists
    try {
        $pdo->query("SELECT 1 FROM event_artists LIMIT 1");
    } catch (Exception $e) {
        sendError('Event-Artist mapping module not initialized. Please run the migration first.', 500);
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    $eventId = isset($_GET['event_id']) ? (int)$_GET['event_id'] : null;
    
    if ($method === 'GET' && $eventId) {
        // Get artists for an event
        if (!hasPermission('organizer.events.view')) {
            sendError('You do not have permission to view events.', 403);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to view this event.', 404);
        }
        
        // Get artists linked to this event
        $stmt = $pdo->prepare("
            SELECT a.* FROM artists a
            INNER JOIN event_artists ea ON a.id = ea.artist_id
            WHERE ea.event_id = :event_id
            AND a.organizer_id = :organizer_id
            AND a.deleted_at IS NULL
            ORDER BY a.name ASC
        ");
        $stmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        $artists = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Decode JSON fields
        foreach ($artists as &$artist) {
            if (!empty($artist['social_links'])) {
                $artist['social_links'] = json_decode($artist['social_links'], true);
            }
        }
        
        sendSuccess($artists);
        
    } elseif ($method === 'POST' && $action === 'link') {
        // Link artist(s) to event
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events.', 403);
        }
        
        $data = getRequestData();
        $eventId = $data['event_id'] ?? null;
        $artistIds = $data['artist_ids'] ?? [];
        
        if (!$eventId) {
            sendError('Event ID is required', 400);
        }
        
        if (!is_array($artistIds) || empty($artistIds)) {
            sendError('At least one artist ID is required', 400);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Verify all artists belong to this organizer
        // Use only positional placeholders to avoid mixing named + positional params
        $placeholders = implode(',', array_fill(0, count($artistIds), '?'));
        $checkArtistsStmt = $pdo->prepare("
            SELECT id FROM artists 
            WHERE id IN ($placeholders) 
              AND organizer_id = ? 
              AND deleted_at IS NULL
        ");
        // Cast all IDs to int to be safe
        $artistIdsInt = array_map('intval', $artistIds);
        $params = array_merge($artistIdsInt, [(int)$organizer['id']]);
        $checkArtistsStmt->execute($params);
        $validArtists = $checkArtistsStmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (count($validArtists) !== count($artistIds)) {
            sendError('One or more artists not found or do not belong to you.', 400);
        }
        
        // Link artists (ignore duplicates)
        $linked = 0;
        $stmt = $pdo->prepare("INSERT IGNORE INTO event_artists (event_id, artist_id) VALUES (:event_id, :artist_id)");
        
        foreach ($artistIds as $artistId) {
            $stmt->execute([':event_id' => $eventId, ':artist_id' => $artistId]);
            if ($stmt->rowCount() > 0) {
                $linked++;
            }
        }
        
        sendSuccess([
            'message' => "Successfully linked $linked artist(s) to event",
            'linked_count' => $linked
        ]);
        
    } elseif ($method === 'POST' && $action === 'unlink') {
        // Unlink artist from event
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events.', 403);
        }
        
        $data = getRequestData();
        $eventId = $data['event_id'] ?? null;
        $artistId = $data['artist_id'] ?? null;
        
        if (!$eventId || !$artistId) {
            sendError('Event ID and Artist ID are required', 400);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Remove link
        $stmt = $pdo->prepare("DELETE FROM event_artists WHERE event_id = :event_id AND artist_id = :artist_id");
        $stmt->execute([':event_id' => $eventId, ':artist_id' => $artistId]);
        
        sendSuccess(['message' => 'Artist unlinked from event successfully']);
        
    } elseif ($method === 'GET' && !$eventId) {
        // Get available artists for organizer (for dropdown)
        if (!hasPermission('organizer.artists.view')) {
            sendError('You do not have permission to view artists.', 403);
        }
        
        $statusFilter = $_GET['status'] ?? 'all';
        
        $sql = "SELECT id, name, profile_image, status FROM artists WHERE organizer_id = :organizer_id AND deleted_at IS NULL";
        $params = [':organizer_id' => $organizer['id']];
        
        if ($statusFilter === 'published') {
            $sql .= " AND status = 'published'";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $artists = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($artists);
        
    } else {
        sendError('Invalid request', 400);
    }
    
} catch (Exception $e) {
    error_log('Event-Artist API Error: ' . $e->getMessage());
    sendError('An error occurred: ' . $e->getMessage(), 500);
}

