<?php
/**
 * Event-Agenda Mapping API
 * Handles linking/unlinking agendas to events
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Check if event_agendas table exists
    try {
        $pdo->query("SELECT 1 FROM event_agendas LIMIT 1");
    } catch (Exception $e) {
        sendError('Event-Agenda mapping module not initialized. Please run the migration first.', 500);
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    $eventId = isset($_GET['event_id']) ? (int)$_GET['event_id'] : null;
    
    if ($method === 'GET' && $eventId) {
        // Get agendas for an event
        if (!hasPermission('organizer.events.view')) {
            sendError('You do not have permission to view events.', 403);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to view this event.', 404);
        }
        
        // Get agendas linked to this event
        $stmt = $pdo->prepare("
            SELECT a.* FROM agendas a
            INNER JOIN event_agendas ea ON a.id = ea.agenda_id
            WHERE ea.event_id = :event_id
              AND a.organizer_id = :organizer_id
              AND a.deleted_at IS NULL
            ORDER BY COALESCE(a.start_time, a.created_at) ASC
        ");
        $stmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        $agendas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($agendas);
        
    } elseif ($method === 'POST' && $action === 'link') {
        // Link agenda(s) to event
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events.', 403);
        }
        
        $data = getRequestData();
        $eventId = $data['event_id'] ?? null;
        $agendaIds = $data['agenda_ids'] ?? [];
        
        if (!$eventId) {
            sendError('Event ID is required', 400);
        }
        
        if (!is_array($agendaIds) || empty($agendaIds)) {
            sendError('At least one agenda ID is required', 400);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Verify all agendas belong to this organizer
        $placeholders = implode(',', array_fill(0, count($agendaIds), '?'));
        $checkAgendasStmt = $pdo->prepare("
            SELECT id FROM agendas 
            WHERE id IN ($placeholders) 
              AND organizer_id = ? 
              AND deleted_at IS NULL
        ");
        $agendaIdsInt = array_map('intval', $agendaIds);
        $params = array_merge($agendaIdsInt, [(int)$organizer['id']]);
        $checkAgendasStmt->execute($params);
        $validAgendas = $checkAgendasStmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (count($validAgendas) !== count($agendaIds)) {
            sendError('One or more agendas not found or do not belong to you.', 400);
        }
        
        // Link agendas (ignore duplicates)
        $linked = 0;
        $stmt = $pdo->prepare("INSERT IGNORE INTO event_agendas (event_id, agenda_id) VALUES (:event_id, :agenda_id)");
        
        foreach ($agendaIds as $agendaId) {
            $stmt->execute([':event_id' => $eventId, ':agenda_id' => $agendaId]);
            if ($stmt->rowCount() > 0) {
                $linked++;
            }
        }
        
        sendSuccess([
            'message' => "Successfully linked $linked agenda(s) to event",
            'linked_count' => $linked
        ]);
        
    } elseif ($method === 'POST' && $action === 'unlink') {
        // Unlink agenda from event
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events.', 403);
        }
        
        $data = getRequestData();
        $eventId = $data['event_id'] ?? null;
        $agendaId = $data['agenda_id'] ?? null;
        
        if (!$eventId || !$agendaId) {
            sendError('Event ID and Agenda ID are required', 400);
        }
        
        // Verify event ownership
        $checkStmt = $pdo->prepare("SELECT id FROM events WHERE id = :event_id AND organizer_id = :organizer_id");
        $checkStmt->execute([':event_id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Remove link
        $stmt = $pdo->prepare("DELETE FROM event_agendas WHERE event_id = :event_id AND agenda_id = :agenda_id");
        $stmt->execute([':event_id' => $eventId, ':agenda_id' => $agendaId]);
        
        sendSuccess(['message' => 'Agenda unlinked from event successfully']);
        
    } elseif ($method === 'GET' && !$eventId) {
        // Get available agendas for organizer (for dropdown)
        if (!hasPermission('organizer.agendas.view')) {
            sendError('You do not have permission to view agendas.', 403);
        }
        
        $statusFilter = $_GET['status'] ?? 'all';
        
        $sql = "SELECT id, title, start_time, end_time, status FROM agendas WHERE organizer_id = :organizer_id AND deleted_at IS NULL";
        $params = [':organizer_id' => $organizer['id']];
        
        if ($statusFilter === 'published') {
            $sql .= " AND status = 'published'";
        }
        
        $sql .= " ORDER BY COALESCE(start_time, created_at) ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $agendas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($agendas);
        
    } else {
        sendError('Invalid request', 400);
    }
    
} catch (Exception $e) {
    error_log('Event-Agenda API Error: ' . $e->getMessage());
    sendError('An error occurred: ' . $e->getMessage(), 500);
}


