<?php
/**
 * Organizer Dashboard API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Get statistics
    $stats = [];
    
    // Total events
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM events WHERE organizer_id = :organizer_id");
    $stmt->execute([':organizer_id' => $organizer['id']]);
    $stats['total_events'] = (int)$stmt->fetch()['total'];
    
    // Total bookings
    $stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT b.id) as total 
        FROM bookings b
        INNER JOIN events e ON b.event_id = e.id
        WHERE e.organizer_id = :organizer_id
    ");
    $stmt->execute([':organizer_id' => $organizer['id']]);
    $stats['total_bookings'] = (int)$stmt->fetch()['total'];
    
    // Total revenue
    $stmt = $pdo->prepare("
        SELECT SUM(b.final_amount) as total 
        FROM bookings b
        INNER JOIN events e ON b.event_id = e.id
        WHERE e.organizer_id = :organizer_id AND b.payment_status = 'completed'
    ");
    $stmt->execute([':organizer_id' => $organizer['id']]);
    $stats['total_revenue'] = (float)($stmt->fetch()['total'] ?? 0);
    
    // Total tickets sold
    $stmt = $pdo->prepare("
        SELECT COUNT(t.id) as total 
        FROM tickets t
        INNER JOIN bookings b ON t.booking_id = b.id
        INNER JOIN events e ON b.event_id = e.id
        WHERE e.organizer_id = :organizer_id
    ");
    $stmt->execute([':organizer_id' => $organizer['id']]);
    $stats['total_tickets_sold'] = (int)$stmt->fetch()['total'];
    
    // Check if deleted_at column exists
    try {
        $checkColumn = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
        $hasDeletedAt = $checkColumn->rowCount() > 0;
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    $deletedAtFilter = $hasDeletedAt ? "AND deleted_at IS NULL" : "";
    
    // Event status counts
    $statusCounts = [];
    $statuses = ['published', 'draft', 'cancelled', 'pending', 'approved', 'live', 'completed'];
    
    foreach ($statuses as $status) {
        $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM events WHERE organizer_id = :organizer_id AND status = :status $deletedAtFilter");
        $stmt->execute([':organizer_id' => $organizer['id'], ':status' => $status]);
        $statusCounts[$status] = (int)$stmt->fetch()['total'];
    }
    
    $stats['published_events'] = $statusCounts['published'] + $statusCounts['approved'] + $statusCounts['live'];
    $stats['draft_events'] = $statusCounts['draft'];
    $stats['cancelled_events'] = $statusCounts['cancelled'];
    $stats['pending_events'] = $statusCounts['pending'];
    
    // Event performance over time (last 14 days)
    $performanceData = [];
    for ($i = 13; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $dateStart = $date . ' 00:00:00';
        $dateEnd = $date . ' 23:59:59';
        
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total 
            FROM events 
            WHERE organizer_id = :organizer_id 
            AND DATE(created_at) = :date
            $deletedAtFilter
        ");
        $stmt->execute([':organizer_id' => $organizer['id'], ':date' => $date]);
        $count = (int)$stmt->fetch()['total'];
        
        $performanceData[] = [
            'date' => date('M j', strtotime($date)),
            'count' => $count
        ];
    }
    
    // Artists stats
    try {
        $pdo->query("SELECT 1 FROM artists LIMIT 1");
        $stmt = $pdo->prepare("SELECT COUNT(*) AS total FROM artists WHERE organizer_id = :organizer_id");
        $stmt->execute([':organizer_id' => $organizer['id']]);
        $stats['total_artists'] = (int)($stmt->fetch()['total'] ?? 0);
        
        $stmt = $pdo->prepare("SELECT COUNT(*) AS total FROM artists WHERE organizer_id = :organizer_id AND status = 'published'");
        $stmt->execute([':organizer_id' => $organizer['id']]);
        $stats['published_artists'] = (int)($stmt->fetch()['total'] ?? 0);
    } catch (Exception $e) {
        $stats['total_artists'] = 0;
        $stats['published_artists'] = 0;
    }
    
    // Sponsors stats
    try {
        $pdo->query("SELECT 1 FROM sponsors LIMIT 1");
        $stmt = $pdo->prepare("SELECT COUNT(*) AS total FROM sponsors WHERE organizer_id = :organizer_id");
        $stmt->execute([':organizer_id' => $organizer['id']]);
        $stats['total_sponsors'] = (int)($stmt->fetch()['total'] ?? 0);
        
        $stmt = $pdo->prepare("SELECT COUNT(*) AS total FROM sponsors WHERE organizer_id = :organizer_id AND status = 'published'");
        $stmt->execute([':organizer_id' => $organizer['id']]);
        $stats['published_sponsors'] = (int)($stmt->fetch()['total'] ?? 0);
    } catch (Exception $e) {
        $stats['total_sponsors'] = 0;
        $stats['published_sponsors'] = 0;
    }
    
    // Agendas stats
    try {
        $pdo->query("SELECT 1 FROM agendas LIMIT 1");
        $stmt = $pdo->prepare("SELECT COUNT(*) AS total FROM agendas WHERE organizer_id = :organizer_id");
        $stmt->execute([':organizer_id' => $organizer['id']]);
        $stats['total_agendas'] = (int)($stmt->fetch()['total'] ?? 0);
        
        $stmt = $pdo->prepare("SELECT COUNT(*) AS total FROM agendas WHERE organizer_id = :organizer_id AND status = 'published'");
        $stmt->execute([':organizer_id' => $organizer['id']]);
        $stats['published_agendas'] = (int)($stmt->fetch()['total'] ?? 0);
    } catch (Exception $e) {
        $stats['total_agendas'] = 0;
        $stats['published_agendas'] = 0;
    }
    
    // Event status distribution for pie chart (Red theme colors)
    $statusDistribution = [
        ['label' => 'Published', 'value' => $stats['published_events'], 'color' => '#dc2626'],
        ['label' => 'Draft', 'value' => $stats['draft_events'], 'color' => '#f87171'],
        ['label' => 'Cancelled', 'value' => $stats['cancelled_events'], 'color' => '#991b1b'],
        ['label' => 'Pending', 'value' => $stats['pending_events'], 'color' => '#fca5a5']
    ];
    
    // Recent events
    $stmt = $pdo->prepare("
        SELECT * FROM events 
        WHERE organizer_id = :organizer_id
        $deletedAtFilter
        ORDER BY created_at DESC
        LIMIT 10
    ");
    $stmt->execute([':organizer_id' => $organizer['id']]);
    $recent_events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccess([
        'stats' => $stats,
        'status_distribution' => $statusDistribution,
        'performance_data' => $performanceData,
        'recent_events' => $recent_events
    ]);
} catch (Exception $e) {
    error_log('Organizer Dashboard API Error: ' . $e->getMessage());
    sendError('Failed to load dashboard data', 500);
}

