<?php
/**
 * Organizer Create Event API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();

    // Soft-delete backward compatibility: deleted_at may not exist if migration wasn't run
    try {
        $check = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
        $hasDeletedAt = ($check && $check->rowCount() > 0);
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    
    if ($method === 'GET') {
        // Get categories and venues for form
        $categoriesStmt = $pdo->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY name ASC");
        $categories = $categoriesStmt->fetchAll(PDO::FETCH_ASSOC);
        
        $venuesStmt = $pdo->query("SELECT id, venue_name, city, country FROM venues WHERE is_active = 1 ORDER BY venue_name ASC");
        $venues = $venuesStmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess([
            'categories' => $categories,
            'venues' => $venues
        ]);
    } elseif ($method === 'POST' && $action !== 'update' && $action !== 'delete') {
        // Create event - Check permission
        if (!hasPermission('organizer.events.create')) {
            sendError('You do not have permission to create events. Please contact your administrator.', 403);
        }
        
        $data = getRequestData();
        $errors = validateRequired($data, ['name', 'start_date', 'end_date']);
        if (!empty($errors)) {
            sendError('Validation failed', 400, $errors);
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO events (
                organizer_id, venue_id, name, description, short_description,
                event_type, category, start_date, end_date, city, state, country,
                address, banner_image, attendees_range, pricing, ticket_available,
                status, language, genre, website_url, refund_policy, age_restriction, created_at
            ) VALUES (
                :organizer_id, :venue_id, :name, :description, :short_description,
                :event_type, :category, :start_date, :end_date, :city, :state, :country,
                :address, :banner_image, :attendees_range, :pricing, :ticket_available,
                'pending', :language, :genre, :website_url, :refund_policy, :age_restriction, NOW()
            )
        ");
        
        $stmt->execute([
            ':organizer_id' => $organizer['id'],
            ':venue_id' => !empty($data['venue_id']) ? (int)$data['venue_id'] : null,
            ':name' => $data['name'],
            ':description' => $data['description'] ?? null,
            ':short_description' => $data['short_description'] ?? null,
            ':event_type' => $data['event_type'] ?? null,
            ':category' => $data['category'] ?? null,
            ':start_date' => $data['start_date'],
            ':end_date' => $data['end_date'],
            ':city' => $data['city'] ?? null,
            ':state' => $data['state'] ?? null,
            ':country' => $data['country'] ?? null,
            ':address' => $data['address'] ?? null,
            ':banner_image' => $data['banner_image'] ?? null,
            ':attendees_range' => $data['attendees_range'] ?? null,
            ':pricing' => $data['pricing'] ?? null,
            ':ticket_available' => isset($data['ticket_available']) ? (int)$data['ticket_available'] : 1,
            ':language' => $data['language'] ?? null,
            ':genre' => $data['genre'] ?? null,
            ':website_url' => $data['website_url'] ?? null,
            ':refund_policy' => $data['refund_policy'] ?? null,
            ':age_restriction' => $data['age_restriction'] ?? null
        ]);
        
        $eventId = $pdo->lastInsertId();
        
        sendSuccess([
            'message' => 'Event created successfully',
            'event_id' => $eventId
        ]);
    } elseif (($method === 'PUT' || $method === 'PATCH' || ($method === 'POST' && $action === 'update')) && isset($_GET['id'])) {
        // Update event - Check permission
        if (!hasPermission('organizer.events.edit')) {
            sendError('You do not have permission to edit events. Please contact your administrator.', 403);
        }
        
        $eventId = (int)$_GET['id'];
        $data = getRequestData();
        
        // Verify event belongs to this organizer (and not deleted if supported)
        $checkSql = "SELECT id FROM events WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or you do not have permission to edit this event.', 404);
        }
        
        // Build update query dynamically based on provided fields
        $updateFields = [];
        $updateParams = [':id' => $eventId];
        
        $allowedFields = ['name', 'description', 'short_description', 'event_type', 'category', 
                         'start_date', 'end_date', 'city', 'state', 'country', 'address', 
                         'banner_image', 'attendees_range', 'pricing', 'ticket_available', 'language', 
                         'genre', 'website_url', 'refund_policy', 'age_restriction'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = $data[$field];
            }
        }
        
        if (empty($updateFields)) {
            sendError('No fields to update', 400);
        }
        
        $updateFields[] = "updated_at = NOW()";
        $sql = "UPDATE events SET " . implode(', ', $updateFields) . " WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        $stmt = $pdo->prepare($sql);
        $updateParams[':organizer_id'] = $organizer['id'];
        $stmt->execute($updateParams);
        
        sendSuccess([
            'message' => 'Event updated successfully',
            'event_id' => $eventId
        ]);
    } elseif (($method === 'DELETE' || ($method === 'POST' && $action === 'delete')) && isset($_GET['id'])) {
        // Delete event - Check permission
        if (!hasPermission('organizer.events.delete')) {
            sendError('You do not have permission to delete events. Please contact your administrator.', 403);
        }
        
        $eventId = (int)$_GET['id'];
        
        // Verify event belongs to this organizer (and not deleted if supported)
        $checkSql = "SELECT id FROM events WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $eventId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Event not found or already deleted.', 404);
        }
        
        if ($hasDeletedAt) {
            // Soft delete event (set deleted_at timestamp)
            $stmt = $pdo->prepare("UPDATE events SET deleted_at = NOW(), status = 'cancelled' WHERE id = :id AND organizer_id = :organizer_id AND deleted_at IS NULL");
            $stmt->execute([':id' => $eventId, ':organizer_id' => $organizer['id']]);
        } else {
            // Fallback: mark as cancelled (best-effort delete without schema change)
            $stmt = $pdo->prepare("UPDATE events SET status = 'cancelled' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $eventId, ':organizer_id' => $organizer['id']]);
        }
        
        if ($stmt->rowCount() === 0) {
            sendError('Event not found or already deleted', 404);
        }
        
        sendSuccess([
            'message' => 'Event deleted successfully'
        ]);
    } else {
        sendError('Method not allowed', 405);
    }
} catch (Exception $e) {
    error_log('Organizer Create Event API Error: ' . $e->getMessage());
    sendError('Failed to process request', 500);
}

