<?php
/**
 * Organizer Attendees API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

// Check permission to view attendees
if (!hasPermission('organizer.attendees.view')) {
    sendError('You do not have permission to view attendees.', 403);
}

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Get filter parameters
    $event_id = $_GET['event_id'] ?? null;
    $search = $_GET['search'] ?? '';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 100);
    $offset = ($page - 1) * $limit;
    
    // Get organizer's events
    $eventsStmt = $pdo->prepare("SELECT id, name, start_date FROM events WHERE organizer_id = :organizer_id ORDER BY start_date DESC");
    $eventsStmt->execute([':organizer_id' => $organizer['id']]);
    $events = $eventsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Build query for attendees
    $where = ["e.organizer_id = :organizer_id"];
    $params = [':organizer_id' => $organizer['id']];
    
    if ($event_id) {
        $where[] = "e.id = :event_id";
        $params[':event_id'] = $event_id;
    }
    
    if ($search) {
        $where[] = "(c.full_name LIKE :search OR c.email LIKE :search OR t.ticket_number LIKE :search)";
        $params[':search'] = '%' . $search . '%';
    }
    
    $whereClause = 'WHERE ' . implode(' AND ', $where);
    
    // Get total count
    $countStmt = $pdo->prepare("
        SELECT COUNT(DISTINCT t.id) as total 
        FROM tickets t
        INNER JOIN bookings b ON t.booking_id = b.id
        INNER JOIN events e ON b.event_id = e.id
        INNER JOIN customers c ON b.customer_id = c.id
        {$whereClause}
    ");
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch()['total'];
    
    // Get attendees
    $stmt = $pdo->prepare("
        SELECT 
            t.*,
            b.booking_reference,
            e.name as event_name,
            e.start_date as event_date,
            c.full_name as customer_name,
            c.email as customer_email,
            c.phone as customer_phone,
            tt.name as ticket_type_name
        FROM tickets t
        INNER JOIN bookings b ON t.booking_id = b.id
        INNER JOIN events e ON b.event_id = e.id
        INNER JOIN customers c ON b.customer_id = c.id
        INNER JOIN ticket_types tt ON t.ticket_type_id = tt.id
        {$whereClause}
        ORDER BY t.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $attendees = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccess([
        'events' => $events,
        'attendees' => $attendees,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Organizer Attendees API Error: ' . $e->getMessage());
    sendError('Failed to load attendees', 500);
}

