<?php
/**
 * Organizer Artists API
 * Handles CRUD operations for artists (organizer-owned only)
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Check if artists table exists
    try {
        $pdo->query("SELECT 1 FROM artists LIMIT 1");
    } catch (Exception $e) {
        sendError('Artists module not initialized. Please run the migration first.', 500);
    }
    
    // Soft-delete backward compatibility
    try {
        $check = $pdo->query("SHOW COLUMNS FROM artists LIKE 'deleted_at'");
        $hasDeletedAt = ($check && $check->rowCount() > 0);
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    $artistId = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if ($method === 'GET') {
        // List artists - only organizer's own artists
        if (!hasPermission('organizer.artists.view')) {
            sendError('You do not have permission to view artists.', 403);
        }
        
        $statusFilter = $_GET['status'] ?? 'all';
        $search = $_GET['search'] ?? '';
        
        $sql = "SELECT * FROM artists WHERE organizer_id = :organizer_id";
        $params = [':organizer_id' => $organizer['id']];
        
        // Exclude soft-deleted
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        // Status filter
        if ($statusFilter !== 'all') {
            $sql .= " AND status = :status";
            $params[':status'] = $statusFilter;
        }
        
        // Search filter
        if (!empty($search)) {
            $sql .= " AND name LIKE :search";
            $params[':search'] = '%' . $search . '%';
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $artists = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Decode JSON fields
        foreach ($artists as &$artist) {
            if (!empty($artist['social_links'])) {
                $artist['social_links'] = json_decode($artist['social_links'], true);
            }
        }
        
        sendSuccess($artists);
        
    } elseif ($method === 'POST' && $action !== 'update' && $action !== 'delete') {
        // Create artist
        if (!hasPermission('organizer.artists.create')) {
            sendError('You do not have permission to create artists.', 403);
        }
        
        $data = getRequestData();
        $errors = validateRequired($data, ['name']);
        if (!empty($errors)) {
            sendError('Validation failed', 400, $errors);
        }
        
        // Prepare social_links JSON
        $socialLinks = null;
        if (!empty($data['social_links'])) {
            if (is_string($data['social_links'])) {
                $socialLinks = $data['social_links'];
            } elseif (is_array($data['social_links']) && !empty(array_filter($data['social_links']))) {
                // Only encode if at least one social link is provided
                $socialLinks = json_encode($data['social_links']);
            }
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO artists (
                organizer_id, name, bio, profile_image, social_links, status, is_active, created_at
            ) VALUES (
                :organizer_id, :name, :bio, :profile_image, :social_links, 'unpublished', 1, NOW()
            )
        ");
        
        $stmt->execute([
            ':organizer_id' => $organizer['id'],
            ':name' => $data['name'],
            ':bio' => $data['bio'] ?? null,
            ':profile_image' => $data['profile_image'] ?? null,
            ':social_links' => $socialLinks
        ]);
        
        $artistId = $pdo->lastInsertId();
        
        sendSuccess([
            'message' => 'Artist created successfully',
            'artist_id' => $artistId
        ]);
        
    } elseif ($method === 'GET' && $artistId) {
        // Get single artist
        if (!hasPermission('organizer.artists.view')) {
            sendError('You do not have permission to view artists.', 403);
        }
        
        $sql = "SELECT * FROM artists WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
        $artist = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$artist) {
            sendError('Artist not found or you do not have permission to view this artist.', 404);
        }
        
        // Decode JSON fields
        if (!empty($artist['social_links'])) {
            $artist['social_links'] = json_decode($artist['social_links'], true);
        }
        
        sendSuccess($artist);
        
    } elseif (($method === 'PUT' || $method === 'PATCH' || ($method === 'POST' && $action === 'update')) && $artistId) {
        // Update artist
        if (!hasPermission('organizer.artists.edit')) {
            sendError('You do not have permission to edit artists.', 403);
        }
        
        // Verify ownership
        $checkSql = "SELECT id FROM artists WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Artist not found or you do not have permission to edit this artist.', 404);
        }
        
        $data = getRequestData();
        
        // Build update query dynamically
        $updateFields = [];
        $updateParams = [
            ':id' => $artistId,
            ':organizer_id' => $organizer['id']
        ];
        
        $allowedFields = ['name', 'bio', 'profile_image', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field]) && $data[$field] !== '') {
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = $data[$field];
            } elseif (isset($data[$field]) && $data[$field] === '' && in_array($field, ['bio', 'profile_image'])) {
                // Allow empty strings for optional fields
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = null;
            }
        }
        
        // Handle social_links separately (JSON)
        if (isset($data['social_links'])) {
            $updateFields[] = "social_links = :social_links";
            if (is_string($data['social_links'])) {
                $updateParams[':social_links'] = $data['social_links'];
            } elseif (is_array($data['social_links']) && !empty(array_filter($data['social_links']))) {
                // Only include if at least one social link is provided
                $updateParams[':social_links'] = json_encode($data['social_links']);
            } else {
                // Empty social links - set to null
                $updateParams[':social_links'] = null;
            }
        }
        
        if (empty($updateFields)) {
            sendError('No fields to update', 400);
        }
        
        $updateFields[] = "updated_at = NOW()";
        $sql = "UPDATE artists SET " . implode(', ', $updateFields) . " WHERE id = :id AND organizer_id = :organizer_id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($updateParams);
        
        sendSuccess(['message' => 'Artist updated successfully']);
        
    } elseif (($method === 'DELETE' || ($method === 'POST' && $action === 'delete')) && $artistId) {
        // Soft delete artist
        if (!hasPermission('organizer.artists.delete')) {
            sendError('You do not have permission to delete artists.', 403);
        }
        
        // Verify ownership
        $checkSql = "SELECT id FROM artists WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Artist not found or you do not have permission to delete this artist.', 404);
        }
        
        if ($hasDeletedAt) {
            // Soft delete
            $stmt = $pdo->prepare("UPDATE artists SET deleted_at = NOW(), status = 'unpublished' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
        } else {
            // Fallback: just set is_active = 0
            $stmt = $pdo->prepare("UPDATE artists SET is_active = 0, status = 'unpublished' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $artistId, ':organizer_id' => $organizer['id']]);
        }
        
        sendSuccess(['message' => 'Artist deleted successfully']);
        
    } else {
        sendError('Invalid request', 400);
    }
    
} catch (Exception $e) {
    error_log('Artist API Error: ' . $e->getMessage());
    sendError('An error occurred: ' . $e->getMessage(), 500);
}

