<?php
/**
 * Organizer Agendas API
 * Handles CRUD operations for agendas (organizer-owned only)
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireOrganizer();

try {
    $organizer = getCurrentOrganizer();
    $pdo = getDB();
    
    // Check if agendas table exists
    try {
        $pdo->query("SELECT 1 FROM agendas LIMIT 1");
    } catch (Exception $e) {
        sendError('Agendas module not initialized. Please run the migration first.', 500);
    }
    
    // Soft-delete backward compatibility
    try {
        $check = $pdo->query("SHOW COLUMNS FROM agendas LIKE 'deleted_at'");
        $hasDeletedAt = ($check && $check->rowCount() > 0);
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    $agendaId = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if ($method === 'GET') {
        // List agendas - only organizer's own agendas
        if (!hasPermission('organizer.agendas.view')) {
            sendError('You do not have permission to view agendas.', 403);
        }
        
        $statusFilter = $_GET['status'] ?? 'all';
        $search = $_GET['search'] ?? '';
        
        $sql = "SELECT * FROM agendas WHERE organizer_id = :organizer_id";
        $params = [':organizer_id' => $organizer['id']];
        
        // Exclude soft-deleted
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        // Status filter
        if ($statusFilter !== 'all') {
            $sql .= " AND status = :status";
            $params[':status'] = $statusFilter;
        }
        
        // Search filter
        if (!empty($search)) {
            $sql .= " AND title LIKE :search";
            $params[':search'] = '%' . $search . '%';
        }
        
        $sql .= " ORDER BY COALESCE(start_time, created_at) ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $agendas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($agendas);
        
    } elseif ($method === 'POST' && $action !== 'update' && $action !== 'delete') {
        // Create agenda
        if (!hasPermission('organizer.agendas.create')) {
            sendError('You do not have permission to create agendas.', 403);
        }
        
        $data = getRequestData();
        $errors = validateRequired($data, ['title']);
        if (!empty($errors)) {
            sendError('Validation failed', 400, $errors);
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO agendas (
                organizer_id, title, description, start_time, end_time, status, is_active, created_at
            ) VALUES (
                :organizer_id, :title, :description, :start_time, :end_time, 'unpublished', 1, NOW()
            )
        ");
        
        $stmt->execute([
            ':organizer_id' => $organizer['id'],
            ':title' => $data['title'],
            ':description' => $data['description'] ?? null,
            ':start_time' => !empty($data['start_time']) ? $data['start_time'] : null,
            ':end_time' => !empty($data['end_time']) ? $data['end_time'] : null,
        ]);
        
        $agendaId = $pdo->lastInsertId();
        
        sendSuccess([
            'message' => 'Agenda created successfully',
            'agenda_id' => $agendaId
        ]);
        
    } elseif ($method === 'GET' && $agendaId) {
        // Get single agenda
        if (!hasPermission('organizer.agendas.view')) {
            sendError('You do not have permission to view agendas.', 403);
        }
        
        $sql = "SELECT * FROM agendas WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $sql .= " AND deleted_at IS NULL";
        }
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
        $agenda = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$agenda) {
            sendError('Agenda not found or you do not have permission to view this agenda.', 404);
        }
        
        sendSuccess($agenda);
        
    } elseif (($method === 'PUT' || $method === 'PATCH' || ($method === 'POST' && $action === 'update')) && $agendaId) {
        // Update agenda
        if (!hasPermission('organizer.agendas.edit')) {
            sendError('You do not have permission to edit agendas.', 403);
        }
        
        // Verify ownership
        $checkSql = "SELECT id FROM agendas WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Agenda not found or you do not have permission to edit this agenda.', 404);
        }
        
        $data = getRequestData();
        
        // Build update query dynamically
        $updateFields = [];
        $updateParams = [
            ':id' => $agendaId,
            ':organizer_id' => $organizer['id']
        ];
        
        $allowedFields = ['title', 'description', 'start_time', 'end_time', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field]) && $data[$field] !== '') {
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = $data[$field];
            } elseif (isset($data[$field]) && $data[$field] === '' && in_array($field, ['description', 'start_time', 'end_time'])) {
                // Allow blank for optional fields (set to NULL)
                $updateFields[] = "{$field} = :{$field}";
                $updateParams[":{$field}"] = null;
            }
        }
        
        if (empty($updateFields)) {
            sendError('No fields to update', 400);
        }
        
        $updateFields[] = "updated_at = NOW()";
        $sql = "UPDATE agendas SET " . implode(', ', $updateFields) . " WHERE id = :id AND organizer_id = :organizer_id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($updateParams);
        
        sendSuccess(['message' => 'Agenda updated successfully']);
        
    } elseif (($method === 'DELETE' || ($method === 'POST' && $action === 'delete')) && $agendaId) {
        // Soft delete agenda
        if (!hasPermission('organizer.agendas.delete')) {
            sendError('You do not have permission to delete agendas.', 403);
        }
        
        // Verify ownership
        $checkSql = "SELECT id FROM agendas WHERE id = :id AND organizer_id = :organizer_id";
        if ($hasDeletedAt) {
            $checkSql .= " AND deleted_at IS NULL";
        }
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
        if (!$checkStmt->fetch()) {
            sendError('Agenda not found or you do not have permission to delete this agenda.', 404);
        }
        
        if ($hasDeletedAt) {
            // Soft delete
            $stmt = $pdo->prepare("UPDATE agendas SET deleted_at = NOW(), status = 'unpublished' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
        } else {
            // Fallback: just set is_active = 0
            $stmt = $pdo->prepare("UPDATE agendas SET is_active = 0, status = 'unpublished' WHERE id = :id AND organizer_id = :organizer_id");
            $stmt->execute([':id' => $agendaId, ':organizer_id' => $organizer['id']]);
        }
        
        sendSuccess(['message' => 'Agenda deleted successfully']);
        
    } else {
        sendError('Invalid request', 400);
    }
    
} catch (Exception $e) {
    error_log('Agenda API Error: ' . $e->getMessage());
    sendError('An error occurred: ' . $e->getMessage(), 500);
}


