<?php
/**
 * LinkedIn OAuth Callback - Handle OAuth response
 */

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/user_activity.php';

// Check for errors
if (isset($_GET['error'])) {
    header('Location: ../../login.html?error=' . urlencode('Authentication failed: ' . $_GET['error']));
    exit;
}

// Verify state token
if (!isset($_GET['state']) || !isset($_SESSION['oauth_state']) || $_GET['state'] !== $_SESSION['oauth_state']) {
    header('Location: ../../login.html?error=' . urlencode('Invalid state token. Please try again.'));
    exit;
}

// Check for authorization code
if (!isset($_GET['code'])) {
    header('Location: ../../login.html?error=' . urlencode('Authorization code not received.'));
    exit;
}

$code = $_GET['code'];

try {
    // Exchange authorization code for access token
    $tokenUrl = 'https://www.linkedin.com/oauth/v2/accessToken';
    $tokenData = [
        'grant_type' => 'authorization_code',
        'code' => $code,
        'redirect_uri' => LINKEDIN_REDIRECT_URI,
        'client_id' => LINKEDIN_CLIENT_ID,
        'client_secret' => LINKEDIN_CLIENT_SECRET
    ];
    
    // LinkedIn secret may contain special characters, use http_build_query which handles encoding
    $ch = curl_init($tokenUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($tokenData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        $errorDetails = json_decode($response, true);
        $errorMessage = 'Failed to exchange authorization code for token';
        if ($errorDetails && isset($errorDetails['error_description'])) {
            $errorMessage .= ': ' . $errorDetails['error_description'];
        } elseif ($curlError) {
            $errorMessage .= ': ' . $curlError;
        } else {
            $errorMessage .= ' (HTTP ' . $httpCode . ')';
        }
        error_log('LinkedIn token exchange error: ' . $errorMessage . ' | Response: ' . $response);
        throw new Exception($errorMessage);
    }
    
    $tokenResponse = json_decode($response, true);
    
    if (!isset($tokenResponse['access_token'])) {
        error_log('LinkedIn token response missing access_token: ' . $response);
        throw new Exception('Access token not received');
    }
    
    $accessToken = $tokenResponse['access_token'];
    
    // Get user info from LinkedIn
    $userInfoUrl = 'https://api.linkedin.com/v2/userinfo';
    
    $ch = curl_init($userInfoUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer ' . $accessToken]);
    
    $userResponse = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception('Failed to get user information from LinkedIn');
    }
    
    $userData = json_decode($userResponse, true);
    
    // Format user data for our system
    $formattedUserData = [
        'sub' => $userData['sub'] ?? $userData['id'] ?? null,
        'email' => $userData['email'] ?? null,
        'name' => trim(($userData['given_name'] ?? '') . ' ' . ($userData['family_name'] ?? '')),
        'given_name' => $userData['given_name'] ?? '',
        'family_name' => $userData['family_name'] ?? '',
        'picture' => $userData['picture'] ?? null,
        'access_token' => $accessToken
    ];
    
    // Use sub as id if available
    if (!$formattedUserData['sub']) {
        throw new Exception('User ID not received from LinkedIn');
    }
    $formattedUserData['id'] = $formattedUserData['sub'];
    
    // Create or update user
    $userId = createOrUpdateUser('linkedin', $formattedUserData);
    
    // Fetch and store user activity (async-like, don't block on errors)
    try {
        fetchLinkedInActivity($accessToken, $userId);
    } catch (Exception $e) {
        error_log('Error fetching LinkedIn activity: ' . $e->getMessage());
        error_log('LinkedIn activity error trace: ' . $e->getTraceAsString());
        // Continue with login even if activity fetch fails
    } catch (Error $e) {
        error_log('Fatal error fetching LinkedIn activity: ' . $e->getMessage());
        error_log('LinkedIn activity fatal error trace: ' . $e->getTraceAsString());
        // Continue with login even if activity fetch fails
    }
    
    // Get full user data
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = :id");
    $stmt->execute([':id' => $userId]);
    $fullUserData = $stmt->fetch();
    
    // Login user
    loginUser($userId, $fullUserData);
    
    // Clear OAuth state
    unset($_SESSION['oauth_state']);
    unset($_SESSION['oauth_provider']);
    
    // Redirect to home
    header('Location: ../../index.html');
    exit;
    
} catch (Exception $e) {
    error_log('LinkedIn OAuth Error: ' . $e->getMessage());
    header('Location: ../../login.html?error=' . urlencode('Authentication failed. Please try again.'));
    exit;
}
