<?php
/**
 * Admin Venues API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';

requireAdmin();

try {
    $pdo = getDB();
    
    // Get filter parameters
    $status = $_GET['status'] ?? 'all';
    $search = $_GET['search'] ?? '';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 100);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = [];
    $params = [];
    
    if ($status !== 'all') {
        if ($status === 'active') {
            $where[] = "v.is_active = 1";
        } elseif ($status === 'inactive') {
            $where[] = "v.is_active = 0";
        } elseif ($status === 'verified') {
            $where[] = "v.verification_status = 'verified'";
        } elseif ($status === 'pending') {
            $where[] = "v.verification_status = 'pending'";
        }
    }
    
    if ($search) {
        $where[] = "(v.venue_name LIKE :search OR v.contact_email LIKE :search OR v.city LIKE :search)";
        $params[':search'] = '%' . $search . '%';
    }
    
    $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
    
    // Get total count
    $countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM venues v {$whereClause}");
    $countStmt->execute($params);
    $totalResult = $countStmt->fetch(PDO::FETCH_ASSOC);
    $total = (int)($totalResult['total'] ?? 0);
    
    // Get venues
    $stmt = $pdo->prepare("
        SELECT 
            v.*,
            c.email as user_email,
            c.full_name as user_name,
            COUNT(DISTINCT e.id) as total_events
        FROM venues v
        LEFT JOIN customers c ON v.user_id = c.id
        LEFT JOIN events e ON v.id = e.venue_id
        {$whereClause}
        GROUP BY v.id
        ORDER BY v.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $venues = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccess([
        'venues' => $venues,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Admin Venues API Error: ' . $e->getMessage());
    sendError('Failed to load venues', 500);
}

