<?php
/**
 * Admin Users API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';

requireAdmin();

try {
    $pdo = getDB();
    
    // Get filter parameters
    $search = $_GET['search'] ?? '';
    $status = $_GET['status'] ?? 'all';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 100);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = [];
    $params = [];
    
    if ($search) {
        $where[] = "(c.email LIKE :search OR c.full_name LIKE :search OR c.first_name LIKE :search OR c.last_name LIKE :search)";
        $params[':search'] = '%' . $search . '%';
    }
    
    if ($status !== 'all') {
        if ($status === 'active') {
            $where[] = "c.is_active = 1";
        } elseif ($status === 'inactive') {
            $where[] = "c.is_active = 0";
        }
    }
    
    $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
    
    // Get total count
    $countStmt = $pdo->prepare("SELECT COUNT(DISTINCT c.id) as total FROM customers c {$whereClause}");
    $countStmt->execute($params);
    $totalResult = $countStmt->fetch(PDO::FETCH_ASSOC);
    $total = (int)($totalResult['total'] ?? 0);
    
    // Get users - first get basic user data
    $stmt = $pdo->prepare("
        SELECT 
            c.id,
            c.email,
            c.first_name,
            c.last_name,
            c.full_name,
            c.phone,
            c.profile_picture,
            c.provider,
            c.provider_id,
            c.is_active,
            c.email_verified,
            c.created_at,
            c.updated_at,
            c.last_login,
            (SELECT COUNT(DISTINCT b.id) FROM bookings b WHERE b.customer_id = c.id) as total_bookings
        FROM customers c
        {$whereClause}
        ORDER BY c.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get user types and roles for each user
    require_once __DIR__ . '/../../includes/permissions.php';
    foreach ($users as &$user) {
        $userId = $user['id'];
        
        // Check if user is admin
        $adminStmt = $pdo->prepare("SELECT role FROM admins WHERE user_id = :user_id AND is_active = 1 LIMIT 1");
        $adminStmt->execute([':user_id' => $userId]);
        $admin = $adminStmt->fetch(PDO::FETCH_ASSOC);
        $user['admin_role'] = $admin['role'] ?? null;
        
        // Check if user is organizer
        $orgStmt = $pdo->prepare("SELECT id FROM organizers WHERE user_id = :user_id AND is_active = 1 LIMIT 1");
        $orgStmt->execute([':user_id' => $userId]);
        $organizer = $orgStmt->fetch(PDO::FETCH_ASSOC);
        $user['organizer_id'] = $organizer['id'] ?? null;
        
        // Check if user is venue
        $venueStmt = $pdo->prepare("SELECT id FROM venues WHERE user_id = :user_id AND is_active = 1 LIMIT 1");
        $venueStmt->execute([':user_id' => $userId]);
        $venue = $venueStmt->fetch(PDO::FETCH_ASSOC);
        $user['venue_id'] = $venue['id'] ?? null;
        
        // Determine user type
        $userType = null;
        if ($user['admin_role']) {
            $userType = 'admin';
        } elseif ($user['organizer_id']) {
            $userType = 'organizer';
        } elseif ($user['venue_id']) {
            $userType = 'venue';
        } else {
            $userType = 'customer';
        }
        $user['user_type'] = $userType;
        
        // Get assigned roles
        try {
            $userRoles = getUserRoles($userId, $userType);
            $user['roles'] = [];
            foreach ($userRoles as $roleSlug) {
                $role = getRoleBySlug($roleSlug);
                if ($role) {
                    $user['roles'][] = $role;
                }
            }
        } catch (Exception $e) {
            // If roles can't be loaded, just set empty array
            error_log('Error loading roles for user ' . $userId . ': ' . $e->getMessage());
            $user['roles'] = [];
        }
    }
    
    sendSuccess([
        'users' => $users,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Admin Users API Error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    sendError('Failed to load users: ' . $e->getMessage(), 500);
}

