<?php
/**
 * User Roles Assignment API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireAdmin();

// Check permission
if (!hasPermission('roles.view') && !isSuperAdmin()) {
    sendError('Insufficient permissions', 403);
}

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$action = $_GET['action'] ?? '';

try {
    $pdo = getDB();
    
    switch ($method) {
        case 'GET':
            if ($action === 'user_roles') {
                // Get roles for a user
                $userId = $_GET['user_id'] ?? null;
                $userType = $_GET['user_type'] ?? null;
                
                if (!$userId) {
                    sendError('User ID is required', 400);
                }
                
                $roles = getUserRoles($userId, $userType);
                $userRoles = [];
                
                foreach ($roles as $roleSlug) {
                    $role = getRoleBySlug($roleSlug);
                    if ($role) {
                        $userRoles[] = $role;
                    } else {
                        // Handle legacy roles that might not be in roles table
                        // Create a temporary role object for display
                        $userRoles[] = [
                            'id' => null,
                            'slug' => $roleSlug,
                            'name' => ucfirst(str_replace('_', ' ', $roleSlug)),
                            'is_system_role' => 1,
                            'is_active' => 1
                        ];
                    }
                }
                
                sendSuccess(['roles' => $userRoles]);
            } elseif ($action === 'users') {
                // Get all users with their roles
                $stmt = $pdo->query("
                    SELECT 
                        c.id,
                        c.email,
                        c.first_name,
                        c.last_name,
                        c.full_name,
                        a.role as admin_role,
                        GROUP_CONCAT(DISTINCT r.name) as custom_roles
                    FROM customers c
                    LEFT JOIN admins a ON c.id = a.user_id AND a.is_active = 1
                    LEFT JOIN user_roles ur ON c.id = ur.user_id
                    LEFT JOIN roles r ON ur.role_id = r.id AND r.is_active = 1
                    WHERE c.is_active = 1
                    GROUP BY c.id
                    ORDER BY c.created_at DESC
                ");
                $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                sendSuccess(['users' => $users]);
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        case 'POST':
            if ($action === 'assign') {
                // Check permission
                if (!hasPermission('roles.edit') && !isSuperAdmin()) {
                    sendError('Insufficient permissions to assign roles', 403);
                }
                
                $data = getRequestData();
                $userId = $data['user_id'] ?? null;
                $roleId = $data['role_id'] ?? null;
                $userType = $data['user_type'] ?? 'admin';
                
                if (!$userId || !$roleId) {
                    sendError('User ID and Role ID are required', 400);
                }
                
                // Validate user_type
                if (!in_array($userType, ['admin', 'organizer', 'venue', 'customer'])) {
                    sendError('Invalid user type', 400);
                }
                
                // Check if user exists
                $stmt = $pdo->prepare("SELECT id FROM customers WHERE id = :id AND is_active = 1");
                $stmt->execute([':id' => $userId]);
                if (!$stmt->fetch()) {
                    sendError('User not found', 404);
                }
                
                // Check if role exists
                $role = getRoleById($roleId);
                if (!$role) {
                    sendError('Role not found', 404);
                }
                
                // Check if user_type column exists
                $checkColumn = $pdo->query("SHOW COLUMNS FROM user_roles LIKE 'user_type'");
                $hasUserType = $checkColumn->rowCount() > 0;
                
                // Check if already assigned (with user_type if column exists)
                if ($hasUserType) {
                    $stmt = $pdo->prepare("
                        SELECT id FROM user_roles 
                        WHERE user_id = :user_id AND role_id = :role_id AND user_type = :user_type
                    ");
                    $stmt->execute([
                        ':user_id' => $userId,
                        ':role_id' => $roleId,
                        ':user_type' => $userType
                    ]);
                } else {
                    $stmt = $pdo->prepare("
                        SELECT id FROM user_roles 
                        WHERE user_id = :user_id AND role_id = :role_id
                    ");
                    $stmt->execute([
                        ':user_id' => $userId,
                        ':role_id' => $roleId
                    ]);
                }
                
                if ($stmt->fetch()) {
                    sendError('Role already assigned to this user', 400);
                }
                
                // Assign role
                if ($hasUserType) {
                    try {
                        $stmt = $pdo->prepare("
                            INSERT INTO user_roles (user_id, role_id, user_type, assigned_by)
                            VALUES (:user_id, :role_id, :user_type, :assigned_by)
                        ");
                        $stmt->execute([
                            ':user_id' => $userId,
                            ':role_id' => $roleId,
                            ':user_type' => $userType,
                            ':assigned_by' => $_SESSION['user_id'] ?? null
                        ]);
                    } catch (PDOException $e) {
                        error_log('Error assigning role: ' . $e->getMessage());
                        error_log('SQL: INSERT INTO user_roles (user_id, role_id, user_type, assigned_by) VALUES (' . $userId . ', ' . $roleId . ', ' . $userType . ', ' . ($_SESSION['user_id'] ?? 'NULL') . ')');
                        throw $e;
                    }
                } else {
                    // If user_type column doesn't exist, insert without it
                    try {
                        $stmt = $pdo->prepare("
                            INSERT INTO user_roles (user_id, role_id, assigned_by)
                            VALUES (:user_id, :role_id, :assigned_by)
                        ");
                        $stmt->execute([
                            ':user_id' => $userId,
                            ':role_id' => $roleId,
                            ':assigned_by' => $_SESSION['user_id'] ?? null
                        ]);
                    } catch (PDOException $e) {
                        error_log('Error assigning role (no user_type): ' . $e->getMessage());
                        throw $e;
                    }
                }
                
                // Verify the assignment was successful
                if ($hasUserType) {
                    $stmt = $pdo->prepare("
                        SELECT ur.*, r.name as role_name 
                        FROM user_roles ur
                        INNER JOIN roles r ON ur.role_id = r.id
                        WHERE ur.user_id = :user_id AND ur.role_id = :role_id AND ur.user_type = :user_type
                    ");
                    $stmt->execute([
                        ':user_id' => $userId,
                        ':role_id' => $roleId,
                        ':user_type' => $userType
                    ]);
                } else {
                    $stmt = $pdo->prepare("
                        SELECT ur.*, r.name as role_name 
                        FROM user_roles ur
                        INNER JOIN roles r ON ur.role_id = r.id
                        WHERE ur.user_id = :user_id AND ur.role_id = :role_id
                    ");
                    $stmt->execute([
                        ':user_id' => $userId,
                        ':role_id' => $roleId
                    ]);
                }
                
                $assigned = $stmt->fetch(PDO::FETCH_ASSOC);
                if (!$assigned) {
                    error_log('Role assignment verification failed for user_id: ' . $userId . ', role_id: ' . $roleId);
                    sendError('Role assignment failed. Please try again.', 500);
                }
                
                error_log('Role assigned successfully: user_id=' . $userId . ', role_id=' . $roleId . ', role_name=' . ($assigned['role_name'] ?? 'N/A') . ', user_type=' . $userType);
                
                sendSuccess(['assignment' => $assigned], 'Role assigned successfully');
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        case 'DELETE':
            if ($action === 'remove') {
                // Check permission
                if (!hasPermission('roles.edit') && !isSuperAdmin()) {
                    sendError('Insufficient permissions to remove roles', 403);
                }
                
                $userId = $_GET['user_id'] ?? null;
                $roleId = $_GET['role_id'] ?? null;
                
                if (!$userId || !$roleId) {
                    sendError('User ID and Role ID are required', 400);
                }
                
                // Remove role
                $stmt = $pdo->prepare("DELETE FROM user_roles WHERE user_id = :user_id AND role_id = :role_id");
                $stmt->execute([':user_id' => $userId, ':role_id' => $roleId]);
                
                sendSuccess([], 'Role removed successfully');
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        default:
            sendError('Method not allowed', 405);
    }
} catch (Exception $e) {
    error_log('User Roles API Error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    sendError('Failed to process request: ' . $e->getMessage(), 500);
}

