<?php
/**
 * Roles Management API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireAdmin();

// Check permission
if (!hasPermission('roles.view') && !isSuperAdmin()) {
    sendError('Insufficient permissions to view roles', 403);
}

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$action = $_GET['action'] ?? '';

try {
    $pdo = getDB();
    
    switch ($method) {
        case 'GET':
            if ($action === 'list') {
                // Get all roles with permission counts
                $roles = getAllRoles();
                
                // Add permission count to each role
                foreach ($roles as &$role) {
                    $permissions = getRolePermissions($role['id']);
                    $role['permission_count'] = count($permissions);
                }
                
                sendSuccess(['roles' => $roles]);
            } elseif ($action === 'get') {
                // Get single role
                $roleId = $_GET['id'] ?? null;
                if (!$roleId) {
                    sendError('Role ID is required', 400);
                }
                
                $role = getRoleById($roleId);
                if (!$role) {
                    sendError('Role not found', 404);
                }
                
                $permissions = getRolePermissions($roleId);
                $role['permissions'] = array_column($permissions, 'id');
                
                sendSuccess(['role' => $role]);
            } elseif ($action === 'permissions') {
                // Get all permissions
                $module = $_GET['module'] ?? null;
                $permissions = getAllPermissions($module);
                sendSuccess(['permissions' => $permissions]);
            } elseif ($action === 'permissions_by_module') {
                // Get permissions grouped by module
                $permissions = getPermissionsByModule();
                sendSuccess(['permissions' => $permissions]);
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        case 'POST':
            if ($action === 'create') {
                // Check permission
                if (!hasPermission('roles.create') && !isSuperAdmin()) {
                    sendError('Insufficient permissions to create roles', 403);
                }
                
                $data = getRequestData();
                $name = $data['name'] ?? '';
                $slug = $data['slug'] ?? '';
                $description = $data['description'] ?? '';
                $permissions = $data['permissions'] ?? [];
                
                if (empty($name) || empty($slug)) {
                    sendError('Name and slug are required', 400);
                }
                
                // Validate slug format
                if (!preg_match('/^[a-z0-9_]+$/', $slug)) {
                    sendError('Slug must contain only lowercase letters, numbers, and underscores', 400);
                }
                
                // Check if slug already exists
                $existing = getRoleBySlug($slug);
                if ($existing) {
                    sendError('Role with this slug already exists', 400);
                }
                
                // Create role
                $stmt = $pdo->prepare("
                    INSERT INTO roles (name, slug, description, is_system_role, is_active)
                    VALUES (:name, :slug, :description, 0, 1)
                ");
                $stmt->execute([
                    ':name' => $name,
                    ':slug' => $slug,
                    ':description' => $description
                ]);
                
                $roleId = $pdo->lastInsertId();
                
                // Assign permissions
                if (!empty($permissions) && is_array($permissions)) {
                    $stmt = $pdo->prepare("
                        INSERT INTO role_permissions (role_id, permission_id)
                        VALUES (:role_id, :permission_id)
                    ");
                    foreach ($permissions as $permissionId) {
                        $stmt->execute([
                            ':role_id' => $roleId,
                            ':permission_id' => $permissionId
                        ]);
                    }
                }
                
                $role = getRoleById($roleId);
                sendSuccess(['role' => $role], 'Role created successfully');
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        case 'PUT':
        case 'PATCH':
            if ($action === 'update') {
                // Check permission
                if (!hasPermission('roles.edit') && !isSuperAdmin()) {
                    sendError('Insufficient permissions to edit roles', 403);
                }
                
                $data = getRequestData();
                $roleId = $data['id'] ?? null;
                
                if (!$roleId) {
                    sendError('Role ID is required', 400);
                }
                
                $role = getRoleById($roleId);
                if (!$role) {
                    sendError('Role not found', 404);
                }
                
                // Check if it's a system role
                if ($role['is_system_role'] && !isSuperAdmin()) {
                    sendError('System roles cannot be modified', 403);
                }
                
                $name = $data['name'] ?? $role['name'];
                $description = $data['description'] ?? $role['description'];
                $isActive = isset($data['is_active']) ? (int)$data['is_active'] : $role['is_active'];
                $permissions = $data['permissions'] ?? null;
                
                // Update role
                $stmt = $pdo->prepare("
                    UPDATE roles 
                    SET name = :name, description = :description, is_active = :is_active
                    WHERE id = :id
                ");
                $stmt->execute([
                    ':name' => $name,
                    ':description' => $description,
                    ':is_active' => $isActive,
                    ':id' => $roleId
                ]);
                
                // Update permissions if provided
                if ($permissions !== null && is_array($permissions)) {
                    // Delete existing permissions
                    $stmt = $pdo->prepare("DELETE FROM role_permissions WHERE role_id = :role_id");
                    $stmt->execute([':role_id' => $roleId]);
                    
                    // Add new permissions
                    if (!empty($permissions)) {
                        $stmt = $pdo->prepare("
                            INSERT INTO role_permissions (role_id, permission_id)
                            VALUES (:role_id, :permission_id)
                        ");
                        foreach ($permissions as $permissionId) {
                            $stmt->execute([
                                ':role_id' => $roleId,
                                ':permission_id' => $permissionId
                            ]);
                        }
                    }
                }
                
                $updatedRole = getRoleById($roleId);
                sendSuccess(['role' => $updatedRole], 'Role updated successfully');
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        case 'DELETE':
            if ($action === 'delete') {
                // Check permission
                if (!hasPermission('roles.delete') && !isSuperAdmin()) {
                    sendError('Insufficient permissions to delete roles', 403);
                }
                
                $roleId = $_GET['id'] ?? null;
                if (!$roleId) {
                    sendError('Role ID is required', 400);
                }
                
                $role = getRoleById($roleId);
                if (!$role) {
                    sendError('Role not found', 404);
                }
                
                // Check if it's a system role
                if ($role['is_system_role']) {
                    sendError('System roles cannot be deleted', 403);
                }
                
                // Check if role is assigned to any users
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM user_roles WHERE role_id = :role_id");
                $stmt->execute([':role_id' => $roleId]);
                $result = $stmt->fetch();
                
                if ($result && $result['count'] > 0) {
                    sendError('Cannot delete role that is assigned to users', 400);
                }
                
                // Delete role (permissions will be deleted via CASCADE)
                $stmt = $pdo->prepare("DELETE FROM roles WHERE id = :id");
                $stmt->execute([':id' => $roleId]);
                
                sendSuccess([], 'Role deleted successfully');
            } else {
                sendError('Invalid action', 400);
            }
            break;
            
        default:
            sendError('Method not allowed', 405);
    }
} catch (Exception $e) {
    error_log('Roles API Error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    sendError('Failed to process request: ' . $e->getMessage(), 500);
}

